<?php

namespace App\Http\Controllers\API\V1\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Otp;
use App\Services\SMSService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    protected $smsService;

    public function __construct(SMSService $smsService)
    {
        $this->smsService = $smsService;
    }

    /**
     * Register new user with OTP verification
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'required|string|unique:users,phone',
            'password' => 'required|string|min:8|confirmed',
            'account_type' => 'sometimes|in:user,business_owner',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // Create user (unverified)
        $user = User::create([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'full_name' => $request->first_name . ' ' . $request->last_name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => Hash::make($request->password),
            'account_type' => $request->account_type ?? 'user',
            'role' => $request->account_type === 'business_owner' ? 'business_owner' : 'user',
            'status' => 'active',
            'phone_verified_at' => null, // Not verified yet
        ]);

        // Generate and send OTP
        $otpCode = $this->smsService->generateOTP();
        $phone = $this->smsService->formatPhone($request->phone);

        Otp::createNew($phone, $otpCode, 'registration');
        $this->smsService->sendOTP($phone, $otpCode);

        return response()->json([
            'success' => true,
            'message' => 'Registration successful. Please verify your phone number with the OTP sent.',
            'data' => [
                'user_id' => $user->id,
                'phone' => $phone,
                'requires_verification' => true,
            ]
        ], 201);
    }

    /**
     * Verify phone number with OTP
     */
    public function verifyOTP(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string',
            'code' => 'required|string|size:6',
            'type' => 'sometimes|in:registration,login,password_reset',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $phone = $this->smsService->formatPhone($request->phone);
        $type = $request->type ?? 'registration';

        $result = Otp::verify($phone, $request->code, $type);

        if (!$result['success']) {
            return response()->json($result, 400);
        }

        // Update user's phone verification
        $user = User::where('phone', $phone)->first();

        if ($user && !$user->phone_verified_at) {
            $user->update(['phone_verified_at' => now()]);
        }

        // Generate token for user
        $token = $user ? $user->createToken('auth-token')->plainTextToken : null;

        return response()->json([
            'success' => true,
            'message' => 'Phone number verified successfully',
            'data' => [
                'user' => $user,
                'token' => $token,
            ]
        ]);
    }

    /**
     * Login with phone and OTP
     */
    public function loginWithOTP(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $phone = $this->smsService->formatPhone($request->phone);

        // Check if user exists
        $user = User::where('phone', $phone)->first();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found with this phone number'
            ], 404);
        }

        if ($user->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Your account is not active. Please contact support.'
            ], 403);
        }

        // Generate and send OTP
        $otpCode = $this->smsService->generateOTP();
        Otp::createNew($phone, $otpCode, 'login');
        $this->smsService->sendOTP($phone, $otpCode);

        return response()->json([
            'success' => true,
            'message' => 'OTP sent to your phone number',
            'data' => [
                'phone' => $phone,
                'requires_otp' => true,
            ]
        ]);
    }

    /**
     * Login with email/phone and password
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'identifier' => 'required|string', // email or phone
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // Determine if identifier is email or phone
        $user = filter_var($request->identifier, FILTER_VALIDATE_EMAIL)
            ? User::where('email', $request->identifier)->first()
            : User::where('phone', $request->identifier)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        if ($user->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Your account is not active'
            ], 403);
        }

        // Update last login
        $user->update(['last_login_at' => now()]);

        // Generate token
        $token = $user->createToken('auth-token')->plainTextToken;

        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'data' => [
                'user' => $user,
                'token' => $token,
            ]
        ]);
    }

    /**
     * Resend OTP
     */
    public function resendOTP(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string',
            'type' => 'sometimes|in:registration,login,password_reset',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $phone = $this->smsService->formatPhone($request->phone);
        $type = $request->type ?? 'login';

        // Generate new OTP
        $otpCode = $this->smsService->generateOTP();
        Otp::createNew($phone, $otpCode, $type);
        $this->smsService->sendOTP($phone, $otpCode);

        return response()->json([
            'success' => true,
            'message' => 'OTP sent successfully'
        ]);
    }

    /**
     * Get current user profile
     */
    public function profile()
    {
        $user = auth()->user()->load(['business', 'subscriptions']);

        return response()->json([
            'success' => true,
            'data' => $user
        ]);
    }

    /**
     * Update user profile
     */
    public function updateProfile(Request $request)
    {
        $user = auth()->user();

        $validator = Validator::make($request->all(), [
            'first_name' => 'sometimes|string|max:255',
            'last_name' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|unique:users,email,' . $user->id,
            'avatar' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $request->only(['first_name', 'last_name', 'email']);

        if ($request->has('first_name') || $request->has('last_name')) {
            $data['full_name'] = ($request->first_name ?? $user->first_name) . ' ' .
                                ($request->last_name ?? $user->last_name);
        }

        if ($request->hasFile('avatar')) {
            $data['avatar'] = $request->file('avatar')->store('avatars', 'public');
        }

        $user->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Profile updated successfully',
            'data' => $user
        ]);
    }

    /**
     * Change password
     */
    public function changePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'current_password' => 'required|string',
            'new_password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $user = auth()->user();

        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect'
            ], 400);
        }

        $user->update([
            'password' => Hash::make($request->new_password)
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully'
        ]);
    }

    /**
     * Logout
     */
    public function logout()
    {
        auth()->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Logged out successfully'
        ]);
    }

    /**
     * Logout from all devices
     */
    public function logoutAll()
    {
        auth()->user()->tokens()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Logged out from all devices successfully'
        ]);
    }
}
