<?php

namespace App\Http\Controllers\API\V1\Business;

use App\Http\Controllers\Controller;
use App\Models\Offer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class OfferManagementController extends Controller
{
    /**
     * Get all offers for authenticated business
     */
    public function index(Request $request)
    {
        $business = auth()->user()->business;

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'You are not associated with any business'
            ], 403);
        }

        $query = Offer::where('business_id', $business->id)
            ->with(['venue', 'category']);

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by approval status
        if ($request->has('approval_status')) {
            $query->where('approval_status', $request->approval_status);
        }

        // Filter by venue
        if ($request->has('venue_id')) {
            $query->where('venue_id', $request->venue_id);
        }

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('title_ar', 'like', "%{$search}%");
            });
        }

        // Sorting
        $sortBy = $request->sort_by ?? 'created_at';
        $sortOrder = $request->sort_order ?? 'desc';
        $query->orderBy($sortBy, $sortOrder);

        $offers = $query->paginate($request->per_page ?? 15);

        return response()->json([
            'success' => true,
            'data' => $offers
        ]);
    }

    /**
     * Create new offer
     */
    public function store(Request $request)
    {
        $business = auth()->user()->business;

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'You are not associated with any business'
            ], 403);
        }

        $validated = $request->validate([
            'venue_id' => 'required|exists:venues,id',
            'category_id' => 'required|exists:categories,id',
            'title' => 'required|string|max:255',
            'title_ar' => 'nullable|string|max:255',
            'description' => 'required|string',
            'description_ar' => 'nullable|string',
            'offer_type' => 'required|in:standard,exclusive,special,flash',
            'discount_type' => 'required|in:percentage,fixed,bogo,free_item',
            'discount_value' => 'required|numeric|min:0',
            'original_price' => 'nullable|numeric|min:0',
            'discounted_price' => 'nullable|numeric|min:0',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'redemption_limit' => 'nullable|integer|min:1',
            'redemption_per_user' => 'required|integer|min:1',
            'terms_conditions' => 'required|string',
            'terms_conditions_ar' => 'nullable|string',
            'images' => 'nullable|array',
            'images.*' => 'image|mimes:jpeg,png,jpg|max:2048',
            'is_featured' => 'boolean',
        ]);

        // Handle image uploads
        $images = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $path = $image->store('offers', 'public');
                $images[] = $path;
            }
        }

        $offer = Offer::create([
            'business_id' => $business->id,
            'venue_id' => $validated['venue_id'],
            'category_id' => $validated['category_id'],
            'title' => $validated['title'],
            'title_ar' => $validated['title_ar'] ?? null,
            'description' => $validated['description'],
            'description_ar' => $validated['description_ar'] ?? null,
            'offer_type' => $validated['offer_type'],
            'discount_type' => $validated['discount_type'],
            'discount_value' => $validated['discount_value'],
            'original_price' => $validated['original_price'] ?? null,
            'discounted_price' => $validated['discounted_price'] ?? null,
            'start_date' => $validated['start_date'],
            'end_date' => $validated['end_date'],
            'redemption_limit' => $validated['redemption_limit'] ?? null,
            'redemption_per_user' => $validated['redemption_per_user'],
            'terms_conditions' => $validated['terms_conditions'],
            'terms_conditions_ar' => $validated['terms_conditions_ar'] ?? null,
            'images' => json_encode($images),
            'is_featured' => $validated['is_featured'] ?? false,
            'status' => 'active',
            'approval_status' => 'pending',
            'qr_code' => Str::random(32),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Offer created successfully and is pending approval',
            'data' => $offer->load(['venue', 'category'])
        ], 201);
    }

    /**
     * Get single offer
     */
    public function show($id)
    {
        $business = auth()->user()->business;

        $offer = Offer::where('business_id', $business->id)
            ->with(['venue', 'category', 'redemptions'])
            ->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $offer
        ]);
    }

    /**
     * Update offer
     */
    public function update(Request $request, $id)
    {
        $business = auth()->user()->business;

        $offer = Offer::where('business_id', $business->id)->findOrFail($id);

        $validated = $request->validate([
            'venue_id' => 'sometimes|exists:venues,id',
            'category_id' => 'sometimes|exists:categories,id',
            'title' => 'sometimes|string|max:255',
            'title_ar' => 'nullable|string|max:255',
            'description' => 'sometimes|string',
            'description_ar' => 'nullable|string',
            'offer_type' => 'sometimes|in:standard,exclusive,special,flash',
            'discount_type' => 'sometimes|in:percentage,fixed,bogo,free_item',
            'discount_value' => 'sometimes|numeric|min:0',
            'original_price' => 'nullable|numeric|min:0',
            'discounted_price' => 'nullable|numeric|min:0',
            'start_date' => 'sometimes|date',
            'end_date' => 'sometimes|date|after:start_date',
            'redemption_limit' => 'nullable|integer|min:1',
            'redemption_per_user' => 'sometimes|integer|min:1',
            'terms_conditions' => 'sometimes|string',
            'terms_conditions_ar' => 'nullable|string',
            'images' => 'nullable|array',
            'images.*' => 'image|mimes:jpeg,png,jpg|max:2048',
            'status' => 'sometimes|in:active,inactive,expired',
            'is_featured' => 'boolean',
        ]);

        // Handle new image uploads
        if ($request->hasFile('images')) {
            $images = [];
            foreach ($request->file('images') as $image) {
                $path = $image->store('offers', 'public');
                $images[] = $path;
            }
            $validated['images'] = json_encode($images);
        }

        $offer->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Offer updated successfully',
            'data' => $offer->load(['venue', 'category'])
        ]);
    }

    /**
     * Delete offer
     */
    public function destroy($id)
    {
        $business = auth()->user()->business;

        $offer = Offer::where('business_id', $business->id)->findOrFail($id);

        // Soft delete
        $offer->delete();

        return response()->json([
            'success' => true,
            'message' => 'Offer deleted successfully'
        ]);
    }

    /**
     * Get offer statistics
     */
    public function statistics($id)
    {
        $business = auth()->user()->business;

        $offer = Offer::where('business_id', $business->id)
            ->with(['redemptions'])
            ->findOrFail($id);

        $stats = [
            'total_views' => $offer->view_count,
            'total_redemptions' => $offer->redemption_count,
            'pending_redemptions' => $offer->redemptions()->where('status', 'pending')->count(),
            'redeemed_count' => $offer->redemptions()->where('status', 'redeemed')->count(),
            'cancelled_count' => $offer->redemptions()->where('status', 'cancelled')->count(),
            'favorite_count' => $offer->favorites()->count(),
            'remaining_redemptions' => $offer->redemption_limit
                ? ($offer->redemption_limit - $offer->redemption_count)
                : null,
            'days_remaining' => now()->diffInDays($offer->end_date, false),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }

    /**
     * Duplicate offer
     */
    public function duplicate($id)
    {
        $business = auth()->user()->business;

        $offer = Offer::where('business_id', $business->id)->findOrFail($id);

        $newOffer = $offer->replicate();
        $newOffer->title = $offer->title . ' (Copy)';
        $newOffer->qr_code = Str::random(32);
        $newOffer->status = 'inactive';
        $newOffer->approval_status = 'pending';
        $newOffer->redemption_count = 0;
        $newOffer->view_count = 0;
        $newOffer->save();

        return response()->json([
            'success' => true,
            'message' => 'Offer duplicated successfully',
            'data' => $newOffer
        ], 201);
    }
}
