<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use Illuminate\Http\Request;

class AuditController extends Controller
{
    /**
     * Display audit logs
     */
    public function index(Request $request)
    {
        $query = AuditLog::with(['user', 'auditable'])->latest();

        if ($request->action) {
            $query->where('action', 'like', "%{$request->action}%");
        }

        if ($request->user_id) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $logs = $query->paginate(50);

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $logs,
            ]);
        }

        return view('admin.audit-logs.index', compact('logs'));
    }

    /**
     * Show single audit log
     */
    public function show($id)
    {
        $log = AuditLog::with(['user', 'auditable'])->findOrFail($id);

        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $log,
            ]);
        }

        return view('admin.audit-logs.show', compact('log'));
    }

    /**
     * Export audit logs
     */
    public function export(Request $request)
    {
        $query = AuditLog::with(['user'])->latest();

        if ($request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $logs = $query->get();

        $filename = 'audit_logs_' . now()->format('Y-m-d_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename={$filename}",
        ];

        $callback = function() use ($logs) {
            $file = fopen('php://output', 'w');

            fputcsv($file, ['ID', 'Action', 'User', 'Email', 'IP Address', 'Date/Time', 'Changes']);

            foreach ($logs as $log) {
                fputcsv($file, [
                    $log->id,
                    $log->action,
                    $log->user ? $log->user->full_name : 'System',
                    $log->user ? $log->user->email : 'N/A',
                    $log->ip_address,
                    $log->created_at->format('Y-m-d H:i:s'),
                    json_encode($log->changes),
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
