<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\User;
use Illuminate\Http\Request;

class BusinessController extends Controller
{
    /**
     * Display business listing
     */
    public function index(Request $request)
    {
        $query = Business::with('owner');

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('business_name', 'like', "%{$search}%")
                    ->orWhere('trade_license_number', 'like', "%{$search}%")
                    ->orWhereHas('owner', function ($q2) use ($search) {
                        $q2->where('full_name', 'like', "%{$search}%");
                    });
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by subscription
        if ($request->filled('subscription')) {
            $query->where('subscription_status', $request->subscription);
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $businesses = $query->paginate(15);

        // Stats
        $stats = [
            'total' => Business::count(),
            'verified' => Business::where('status', 'verified')->count(),
            'pending' => Business::where('status', 'pending')->count(),
            'suspended' => Business::where('status', 'suspended')->count(),
        ];

        return view('admin.businesses.index', compact('businesses', 'stats'));
    }

    /**
     * Show business details
     */
    public function show($id)
    {
        $business = Business::with(['owner', 'venues', 'offers'])->findOrFail($id);
        return view('admin.businesses.show', compact('business'));
    }

    /**
     * Show business verification page
     */
    public function showVerify($id)
    {
        $business = Business::with(['owner', 'venues', 'offers'])->findOrFail($id);
        return view('admin.businesses.verify', compact('business'));
    }

    /**
     * Verify business
     */
    public function verify(Request $request, $id)
    {
        $business = Business::findOrFail($id);
// Convert checkbox to boolean
        $request->merge([
            'send_email' => $request->has('send_email'),
        ]);

        $validated = $request->validate([
            'action' => 'required|in:approve,reject',
            'notes' => 'nullable|string',
            'send_email' => 'boolean',
        ]);

        if ($validated['action'] === 'approve') {
            $business->update([
                'status' => 'verified',
                'verified_at' => now(),
                'verification_notes' => $validated['notes'] ?? null,
            ]);

            $message = 'Business verified successfully.';

            // TODO: Send verification email if send_email is true

        } else {
            $business->update([
                'status' => 'rejected',
                'verification_notes' => $validated['notes'] ?? null,
            ]);

            $message = 'Business rejected.';

            // TODO: Send rejection email if send_email is true
        }

        return redirect()->route('admin.businesses.index')
            ->with('success', $message);
    }

    /**
     * Approve business
     */
    public function approve($id)
    {
        $business = Business::findOrFail($id);

        $business->update([
            'status' => 'verified',
            'verified_at' => now()
        ]);

        // Create notification
        \App\Models\Notification::create([
            'notifiable_type' => 'App\\Models\\User',
            'notifiable_id' => $business->owner_id,
            'type' => 'business_approved',
            'data' => json_encode([
                'title' => 'Business Approved',
                'message' => "Your business '{$business->business_name}' has been approved and verified.",
                'business_id' => $business->id
            ])
        ]);

        // Log audit
        \App\Models\AuditLog::logAction(
            'approve_business',
            $business,
            null,
            ['status' => 'approved']
        );

        return response()->json([
            'success' => true,
            'message' => 'Business approved successfully'
        ]);
    }

    /**
     * Reject business
     */
    public function reject(Request $request, $id)
    {
        $request->validate([
            'reason' => 'required|string'
        ]);

        $business = Business::findOrFail($id);

        $business->update([
            'status' => 'rejected',
            'verification_notes' => $request->reason
        ]);

        // Create notification
        \App\Models\Notification::create([
            'notifiable_type' => 'App\\Models\\User',
            'notifiable_id' => $business->owner_id,
            'type' => 'business_rejected',
            'data' => json_encode([
                'title' => 'Business Rejected',
                'message' => "Your business '{$business->business_name}' has been rejected. Reason: {$request->reason}",
                'business_id' => $business->id
            ])
        ]);

        // Log audit
        \App\Models\AuditLog::logAction(
            'reject_business',
            $business,
            null,
            ['status' => 'rejected', 'reason' => $request->reason]
        );

        return response()->json([
            'success' => true,
            'message' => 'Business rejected successfully'
        ]);
    }

    /**
     * Suspend business
     */
    public function suspend(Request $request, $id)
    {
        $request->validate([
            'reason' => 'required|string'
        ]);

        $business = Business::findOrFail($id);

        $business->update([
            'status' => 'suspended',
            'verification_notes' => $request->reason,
        ]);

        // Create notification
        \App\Models\Notification::create([
            'notifiable_type' => 'App\\Models\\User',
            'notifiable_id' => $business->owner_id,
            'type' => 'business_suspended',
            'data' => json_encode([
                'title' => 'Business Suspended',
                'message' => "Your business '{$business->business_name}' has been suspended. Reason: {$request->reason}",
                'business_id' => $business->id
            ])
        ]);

        // Log audit
        \App\Models\AuditLog::logAction(
            'suspend_business',
            $business,
            null,
            ['status' => 'suspended', 'reason' => $request->reason]
        );

        return response()->json([
            'success' => true,
            'message' => 'Business suspended successfully'
        ]);
    }

    /**
     * Activate business
     */
    public function activate($id)
    {
        $business = Business::findOrFail($id);

        $business->update([
            'status' => 'verified',
        ]);

        // Create notification
        \App\Models\Notification::create([
            'notifiable_type' => 'App\\Models\\User',
            'notifiable_id' => $business->owner_id,
            'type' => 'business_activated',
            'data' => json_encode([
                'title' => 'Business Activated',
                'message' => "Your business '{$business->business_name}' has been activated.",
                'business_id' => $business->id
            ])
        ]);

        // Log audit
        \App\Models\AuditLog::logAction(
            'activate_business',
            $business,
            null,
            ['status' => 'activated']
        );

        return response()->json([
            'success' => true,
            'message' => 'Business activated successfully'
        ]);
    }

    /**
     * Delete business
     */
    public function destroy($id)
    {
        $business = Business::findOrFail($id);

        // Check if business has active subscriptions or venues
        if ($business->venues()->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete business with existing venues'
            ], 400);
        }

        // Log audit
        \App\Models\AuditLog::logAction(
            'delete_business',
            $business,
            null,
            ['business_name' => $business->business_name]
        );

        $business->delete();

        return response()->json([
            'success' => true,
            'message' => 'Business deleted successfully'
        ]);
    }

    /**
     * Export businesses
     */
    public function export()
    {
        $businesses = Business::with('owner')->get();

        $csvData = [];
        $csvData[] = [
            'ID', 'Business Name', 'Owner', 'License Number',
            'Email', 'Phone', 'Status', 'Subscription', 'Joined'
        ];

        foreach ($businesses as $business) {
            $csvData[] = [
                $business->id,
                $business->business_name,
                $business->owner->full_name ?? 'N/A',
                $business->trade_license_number,
                $business->email,
                $business->phone,
                $business->status,
                $business->subscription_status,
                $business->created_at->format('Y-m-d'),
            ];
        }

        $filename = 'businesses_' . date('Y-m-d') . '.csv';

        $handle = fopen('php://temp', 'r+');
        foreach ($csvData as $row) {
            fputcsv($handle, $row);
        }
        rewind($handle);
        $csv = stream_get_contents($handle);
        fclose($handle);

        return response($csv)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    /**
     * Assign subscription to business
     */
    public function assignSubscription(Request $request, $id)
    {
        $validated = $request->validate([
            'plan_id' => 'required|exists:subscription_plans,id',
            'duration' => 'required|integer|min:1|max:3650'
        ]);

        $business = Business::findOrFail($id);
        $plan = \App\Models\SubscriptionPlan::findOrFail($validated['plan_id']);

        // Calculate dates (duration is in days)
        $startDate = now();
        $endDate = now()->addDays($validated['duration']);

        // Cancel existing active subscriptions
        $business->subscriptions()
            ->where('status', 'active')
            ->update(['status' => 'cancelled', 'cancelled_at' => now()]);

        // Create new subscription
        $subscription = \App\Models\Subscription::create([
            'user_id' => $business->owner_id,
            'business_id' => $business->id,
            'subscription_plan_id' => $plan->id,
            'status' => 'active',
            'start_date' => $startDate,
            'end_date' => $endDate,
            'auto_renew' => false
        ]);

        // Determine if free plan (price = 0)
        $isFree = $plan->price == 0;

        // Create payment record
        \App\Models\Payment::create([
            'user_id' => $business->owner_id,
            'business_id' => $business->id,
            'subscription_id' => $subscription->id,
            'amount' => 0,
            'currency' => $plan->currency,
            'payment_method' => $isFree ? 'free' : 'admin_assigned',
            'status' => 'completed',
            'type' => 'subscription',
            'payment_date' => now(),
            'metadata' => json_encode([
                'assigned_by_admin' => auth()->id(),
                'is_free_plan' => $isFree,
                'duration_days' => $validated['duration'],
                'plan_price' => $plan->price
            ])
        ]);

        // Update business profile
        $business->update([
            'subscription_plan_id' => $plan->id,
            'subscription_status' => 'active',
            'subscription_start_date' => $startDate,
            'subscription_end_date' => $endDate,
        ]);

        // Create notification
        \App\Models\Notification::create([
            'notifiable_type' => 'App\\Models\\User',
            'notifiable_id' => $business->owner_id,
            'type' => 'subscription_assigned',
            'data' => json_encode([
                'title' => 'Subscription Assigned',
                'message' => "A {$plan->name} subscription has been assigned to your business for {$validated['duration']} day(s).",
                'plan_name' => $plan->name,
                'duration_days' => $validated['duration'],
                'is_free' => $isFree
            ])
        ]);

        // Log audit
        \App\Models\AuditLog::logAction(
            'assign_subscription',
            $business,
            null,
            [
                'plan_id' => $plan->id,
                'plan_name' => $plan->name,
                'plan_price' => $plan->price,
                'duration_days' => $validated['duration'],
                'is_free_plan' => $isFree,
                'admin_id' => auth()->id()
            ]
        );

//        if plan has team management then assign permission to the business owner
        $user = User::find($business->owner_id);
//       add in the table business_staff if plan has permission about team
        if ($plan->max_staff >= 1 ?? false) {
            \App\Models\BusinessStaff::updateOrCreate(
                [
                    'business_id' => $business->id,
                    'user_id' => $user->id,
                ],
                [
                    'role' => 'owner',
                    'permissions' => ['all'],
                    'status' => 'active',
                    'joined_at' => now(),
                ]
            );
        }


        return redirect()->back()->with('success', 'Subscription assigned successfully!');
    }
}
