<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Business;
use App\Models\Venue;
use App\Models\Offer;
use App\Models\Subscription;
use App\Models\Payment;
use App\Models\Redemption;
use App\Models\Spotlight;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        // System Overview
        $overview = [
            'total_users' => User::where('account_type', 'user')->count(),
            'active_users' => User::where('account_type', 'user')
                ->where('status', 'active')
                ->count(),
            'new_users_today' => User::where('account_type', 'user')
                ->whereDate('created_at', today())
                ->count(),

            'total_businesses' => Business::count(),
            'verified_businesses' => Business::where('status', 'verified')->count(),
            'pending_verification' => Business::where('status', 'pending')->count(),

            'total_venues' => Venue::count(),
            'active_venues' => Venue::where('status', 'active')->count(),

            'total_offers' => Offer::count(),
            'active_offers' => Offer::where('status', 'active')->count(),
            'pending_approval' => Offer::where('approval_status', 'pending')->count(),

            'total_subscriptions' => Subscription::count(),
            'active_subscriptions' => Subscription::where('status', 'active')->count(),

            'total_redemptions' => Redemption::count(),
            'pending_redemptions' => Redemption::where('status', 'pending')->count(),
            'redeemed_today' => Redemption::where('status', 'redeemed')
                ->whereDate('redeemed_at', today())
                ->count(),

            'total_spotlights' => Spotlight::count(),
            'pending_moderation' => Spotlight::where('moderation_status', 'pending')->count(),
            'flagged_content' => Spotlight::where('status', 'flagged')->count(),
        ];

        // Revenue Stats
        $revenue = [
            'total_revenue' => Payment::where('status', 'completed')->sum('amount'),
            'revenue_today' => Payment::where('status', 'completed')
                ->whereDate('payment_date', today())
                ->sum('amount'),
            'revenue_this_month' => Payment::where('status', 'completed')
                ->whereMonth('payment_date', now()->month)
                ->whereYear('payment_date', now()->year)
                ->sum('amount'),
            'pending_payments' => Payment::where('status', 'pending')->count(),
            'failed_payments' => Payment::where('status', 'failed')->count(),
        ];

        // Recent Activity
        $recentUsers = User::where('account_type', 'user')
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        $recentBusinesses = Business::with('owner')
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        $pendingBusinesses = Business::with('owner')
            ->where('status', 'pending')
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        $pendingOffers = Offer::with(['business', 'venue'])
            ->where('approval_status', 'pending')
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        $pendingSpotlights = Spotlight::with('user')
            ->where('moderation_status', 'pending')
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        // Charts Data (Last 30 Days)
        $thirtyDaysAgo = now()->subDays(30);

        $userGrowth = User::where('account_type', 'user')
            ->where('created_at', '>=', $thirtyDaysAgo)
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as count'))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $revenueGrowth = Payment::where('status', 'completed')
            ->where('payment_date', '>=', $thirtyDaysAgo)
            ->select(
                DB::raw('DATE(payment_date) as date'),
                DB::raw('SUM(amount) as total')
            )
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $redemptionsGrowth = Redemption::where('created_at', '>=', $thirtyDaysAgo)
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as count'))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Top Performers
        $topBusinesses = Business::where('status', 'verified')
            ->orderBy('rating', 'desc')
            ->orderBy('total_reviews', 'desc')
            ->take(10)
            ->get();

        $topVenues = Venue::where('status', 'active')
            ->orderBy('rating', 'desc')
            ->orderBy('view_count', 'desc')
            ->take(10)
            ->get();

        $topOffers = Offer::where('status', 'active')
            ->orderBy('redemption_count', 'desc')
            ->orderBy('view_count', 'desc')
            ->take(10)
            ->get();

        return view('admin.dashboard', compact(
            'overview',
            'revenue',
            'recentUsers',
            'recentBusinesses',
            'pendingBusinesses',
            'pendingOffers',
            'pendingSpotlights',
            'userGrowth',
            'revenueGrowth',
            'redemptionsGrowth',
            'topBusinesses',
            'topVenues',
            'topOffers'
        ));
    }

    public function analytics(Request $request)
    {
        $days = $request->input('days', 30);
        $startDate = now()->subDays($days);

        // User Analytics
        $userStats = [
            'total' => User::where('account_type', 'user')->count(),
            'active' => User::where('account_type', 'user')->where('status', 'active')->count(),
            'new' => User::where('account_type', 'user')->where('created_at', '>=', $startDate)->count(),
            'growth' => User::where('account_type', 'user')
                ->where('created_at', '>=', $startDate)
                ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as count'))
                ->groupBy('date')
                ->orderBy('date')
                ->get(),
        ];

        // Business Analytics
        $businessStats = [
            'total' => Business::count(),
            'verified' => Business::where('status', 'verified')->count(),
            'new' => Business::where('created_at', '>=', $startDate)->count(),
            'growth' => Business::where('created_at', '>=', $startDate)
                ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as count'))
                ->groupBy('date')
                ->orderBy('date')
                ->get(),
        ];

        // Revenue Analytics
        $revenueStats = [
            'total' => Payment::where('status', 'completed')->sum('amount'),
            'period' => Payment::where('status', 'completed')
                ->where('payment_date', '>=', $startDate)
                ->sum('amount'),
            'growth' => Payment::where('status', 'completed')
                ->where('payment_date', '>=', $startDate)
                ->select(
                    DB::raw('DATE(payment_date) as date'),
                    DB::raw('SUM(amount) as total')
                )
                ->groupBy('date')
                ->orderBy('date')
                ->get(),
        ];

        // Redemption Analytics
        $redemptionStats = [
            'total' => Redemption::count(),
            'period' => Redemption::where('created_at', '>=', $startDate)->count(),
            'growth' => Redemption::where('created_at', '>=', $startDate)
                ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as count'))
                ->groupBy('date')
                ->orderBy('date')
                ->get(),
        ];

        return view('admin.analytics', compact(
            'userStats',
            'businessStats',
            'revenueStats',
            'redemptionStats',
            'days'
        ));
    }

    public function systemHealth()
    {
        $health = [
            'database' => $this->checkDatabase(),
            'redis' => $this->checkRedis(),
            'storage' => $this->checkStorage(),
            'queue' => $this->checkQueue(),
        ];

        $allHealthy = collect($health)->every(fn($status) => $status === 'ok');

        return response()->json([
            'success' => true,
            'data' => [
                'status' => $allHealthy ? 'healthy' : 'degraded',
                'checks' => $health,
            ]
        ]);
    }

    protected function checkDatabase()
    {
        try {
            DB::connection()->getPdo();
            return 'ok';
        } catch (\Exception $e) {
            return 'error';
        }
    }

    protected function checkRedis()
    {
        try {
            \Illuminate\Support\Facades\Redis::ping();
            return 'ok';
        } catch (\Exception $e) {
            return 'error';
        }
    }

    protected function checkStorage()
    {
        try {
            $diskSpace = disk_free_space('/');
            $diskTotal = disk_total_space('/');
            $percentFree = ($diskSpace / $diskTotal) * 100;

            return $percentFree > 10 ? 'ok' : 'warning';
        } catch (\Exception $e) {
            return 'error';
        }
    }

    protected function checkQueue()
    {
        try {
            // Check if queue is running
            // This is a simple check - implement proper queue monitoring
            return 'ok';
        } catch (\Exception $e) {
            return 'error';
        }
    }
}
