<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\Faq;
use Illuminate\Http\Request;

class FaqController extends Controller
{
    /**
     * List all FAQs
     */
    public function index(Request $request)
    {
        $query = Faq::query();

        if ($request->category) {
            $query->where('category', $request->category);
        }

        if ($request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('question', 'LIKE', "%{$search}%")
                    ->orWhere('answer', 'LIKE', "%{$search}%");
            });
        }

        $faqs = $query->orderBy('category')->orderBy('order')->get();

        return response()->json([
            'success' => true,
            'data' => $faqs,
        ]);
    }

    /**
     * Get single FAQ
     */
    public function show($id)
    {
        $faq = Faq::findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $faq,
        ]);
    }

    /**
     * Create FAQ
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'question' => 'required|string|max:500',
            'question_ar' => 'nullable|string|max:500',
            'answer' => 'required|string',
            'answer_ar' => 'nullable|string',
            'category' => 'required|in:general,account,subscription,offers,business,payment,privacy',
            'order' => 'integer|min:0',
            'is_active' => 'boolean',
        ]);

        $faq = Faq::create($validated);

        AuditLog::logAction('faq.created', $faq);

        return response()->json([
            'success' => true,
            'message' => 'FAQ created successfully',
            'data' => $faq,
        ], 201);
    }

    /**
     * Update FAQ
     */
    public function update(Request $request, $id)
    {
        $faq = Faq::findOrFail($id);

        $validated = $request->validate([
            'question' => 'required|string|max:500',
            'question_ar' => 'nullable|string|max:500',
            'answer' => 'required|string',
            'answer_ar' => 'nullable|string',
            'category' => 'required|in:general,account,subscription,offers,business,payment,privacy',
            'order' => 'integer|min:0',
            'is_active' => 'boolean',
        ]);

        $old = $faq->toArray();
        $faq->update($validated);

        AuditLog::logAction('faq.updated', $faq, $old, $faq->toArray());

        return response()->json([
            'success' => true,
            'message' => 'FAQ updated successfully',
            'data' => $faq,
        ]);
    }

    /**
     * Delete FAQ
     */
    public function destroy($id)
    {
        $faq = Faq::findOrFail($id);

        AuditLog::logAction('faq.deleted', $faq, $faq->toArray());

        $faq->delete();

        return response()->json([
            'success' => true,
            'message' => 'FAQ deleted successfully',
        ]);
    }

    /**
     * Toggle FAQ status
     */
    public function toggle($id)
    {
        $faq = Faq::findOrFail($id);

        $faq->update(['is_active' => !$faq->is_active]);

        AuditLog::logAction('faq.toggled', $faq);

        return response()->json([
            'success' => true,
            'message' => 'FAQ status updated',
            'data' => $faq,
        ]);
    }

    /**
     * Reorder FAQs
     */
    public function reorder(Request $request)
    {
        $validated = $request->validate([
            'faqs' => 'required|array',
            'faqs.*.id' => 'required|exists:faqs,id',
            'faqs.*.order' => 'required|integer|min:0',
        ]);

        foreach ($validated['faqs'] as $item) {
            Faq::where('id', $item['id'])->update(['order' => $item['order']]);
        }

        AuditLog::logAction('faq.reordered', null, null, null, ['count' => count($validated['faqs'])]);

        return response()->json([
            'success' => true,
            'message' => 'FAQs reordered successfully',
        ]);
    }
}
