<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Role;
use App\Models\Permission;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class RoleController extends Controller
{
    public function index(Request $request)
    {
        $type = $request->get('type', 'all');

        $query = Role::with('permissions');

        if ($type !== 'all') {
            $query->where('type', $type);
        }

        $roles = $query->orderBy('created_at', 'desc')->paginate(20);

        $stats = [
            'total' => Role::count(),
            'admin' => Role::where('type', 'admin')->count(),
            'business' => Role::where('type', 'business')->count(),
            'system' => Role::where('is_system', true)->count(),
        ];

        return view('admin.roles.index', compact('roles', 'stats', 'type'));
    }

    public function create()
    {
        $permissions = Permission::grouped();

        return view('admin.roles.create', compact('permissions'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:roles,name',
            'display_name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:admin,business',
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role = Role::create([
            'name' => Str::slug($validated['name'], '_'),
            'display_name' => $validated['display_name'],
            'description' => $validated['description'] ?? null,
            'type' => $validated['type'],
            'is_system' => false,
        ]);

        if (!empty($validated['permissions'])) {
            $role->permissions()->attach($validated['permissions']);
        }

        return redirect()
            ->route('admin.roles.index')
            ->with('success', 'Role created successfully');
    }

    public function show($id)
    {
        $role = Role::with(['permissions', 'users'])->findOrFail($id);

        $userCount = $role->users()->count();
        $permissionCount = $role->permissions()->count();

        return view('admin.roles.show', compact('role', 'userCount', 'permissionCount'));
    }

    public function edit($id)
    {
        $role = Role::with('permissions')->findOrFail($id);

        if ($role->is_system) {
            return redirect()
                ->route('admin.roles.index')
                ->with('error', 'System roles cannot be edited');
        }

        $permissions = Permission::grouped();
        $rolePermissionIds = $role->permissions->pluck('id')->toArray();

        return view('admin.roles.edit', compact('role', 'permissions', 'rolePermissionIds'));
    }

    public function update(Request $request, $id)
    {
        $role = Role::findOrFail($id);

        if ($role->is_system) {
            return response()->json([
                'success' => false,
                'message' => 'System roles cannot be modified'
            ], 403);
        }

        $validated = $request->validate([
            'display_name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role->update([
            'display_name' => $validated['display_name'],
            'description' => $validated['description'] ?? null,
        ]);

        $role->permissions()->sync($validated['permissions'] ?? []);

        return redirect()
            ->route('admin.roles.index')
            ->with('success', 'Role updated successfully');
    }

    public function destroy($id)
    {
        $role = Role::findOrFail($id);

        if ($role->is_system) {
            return response()->json([
                'success' => false,
                'message' => 'System roles cannot be deleted'
            ], 403);
        }

        $userCount = $role->users()->count();

        if ($userCount > 0) {
            return response()->json([
                'success' => false,
                'message' => "Cannot delete role. {$userCount} users are assigned to this role."
            ], 400);
        }

        $role->permissions()->detach();
        $role->delete();

        return redirect()
            ->route('admin.roles.index')
            ->with('success', 'Role deleted successfully');
    }

    public function permissions($id)
    {
        $role = Role::with('permissions')->findOrFail($id);
        $allPermissions = Permission::grouped();

        return view('admin.roles.permissions', compact('role', 'allPermissions'));
    }

    public function updatePermissions(Request $request, $id)
    {
        $role = Role::findOrFail($id);


        $validated = $request->validate([
            'permissions' => 'required|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role->permissions()->sync($validated['permissions']);

        return redirect()
            ->route('admin.roles.permissions', $role->id)
            ->with('success', 'Permissions updated successfully');
    }

    public function users($id)
    {
        $role = Role::findOrFail($id);
        $users = $role->users()->with('businesses')->paginate(20);

        return view('admin.roles.users', compact('role', 'users'));
    }
}
