<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Spotlight;
use Illuminate\Http\Request;

class SpotlightController extends Controller
{
    /**
     * Display spotlight listing
     */
    public function index(Request $request)
    {
        $query = Spotlight::with(['user', 'venue']);

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('content', 'like', "%{$search}%")
                    ->orWhereHas('user', function($q2) use ($search) {
                        $q2->where('full_name', 'like', "%{$search}%");
                    });
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by flagged
        if ($request->has('flagged')) {
            $query->where('status', 'flagged');
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $spotlights = $query->paginate(20);

        // Stats
        $stats = [
            'total' => Spotlight::count(),
            'active' => Spotlight::where('status', 'approved')->count(),
            'pending' => Spotlight::where('status', 'pending')->count(),
            'flagged' => Spotlight::where('status', 'flagged')->count(),
        ];

        return view('admin.spotlights.index', compact('spotlights', 'stats'));
    }

    /**
     * Show spotlight details
     */
    public function show($id)
    {
        $spotlight = Spotlight::with([
            'user',
            'venue',
            'likes',
            'comments.user'
        ])->findOrFail($id);

        return view('admin.spotlights.show', compact('spotlight'));
    }

    /**
     * Approve spotlight
     */
    public function approve($id)
    {
        $spotlight = Spotlight::findOrFail($id);

        $spotlight->update([
            'status' => 'active',
            'moderation_notes' => request('notes'),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Spotlight approved successfully'
        ]);
    }

    /**
     * Reject spotlight
     */
    public function reject($id)
    {
        $spotlight = Spotlight::findOrFail($id);

        $spotlight->update([
            'status' => 'rejected',
            'moderation_notes' => request('notes'),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Spotlight rejected'
        ]);
    }

    /**
     * Delete spotlight
     */
    public function destroy($id)
    {
        $spotlight = Spotlight::findOrFail($id);
        $spotlight->delete();

        return response()->json([
            'success' => true,
            'message' => 'Spotlight deleted successfully'
        ]);
    }

    /**
     * Unflag spotlight
     */
    public function unflag($id)
    {
        $spotlight = Spotlight::findOrFail($id);

        $spotlight->update([
            'status' => 'approved',
            'moderation_status' => 'approved',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Flag removed'
        ]);
    }
}
