<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Venue;
use App\Models\Business;
use App\Models\Category;
use Illuminate\Http\Request;

class VenueController extends Controller
{
    /**
     * Display venue listing
     */
    public function index(Request $request)
    {
        $query = Venue::with(['business', 'category']);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('city', 'like', "%{$search}%")
                    ->orWhereHas('business', function($q2) use ($search) {
                        $q2->where('business_name', 'like', "%{$search}%");
                    });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->filled('city')) {
            $query->where('city', $request->city);
        }

        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $venues = $query->paginate(20);

        $stats = [
            'total' => Venue::count(),
            'active' => Venue::where('status', 'active')->count(),
            'inactive' => Venue::where('status', 'inactive')->count(),
            'featured' => Venue::where('is_featured', true)->count(),
        ];

        $categories = Category::all();
        $cities = Venue::distinct('city')->pluck('city');

        return view('admin.venues.index', compact('venues', 'stats', 'categories', 'cities'));
    }

    /**
     * Show venue details
     */
    public function show($id)
    {
        $venue = Venue::with([
            'business',
            'category',
            'offers',
            'reviews.user'
        ])->findOrFail($id);

        return view('admin.venues.show', compact('venue'));
    }

    /**
     * Approve venue
     */
    public function approve($id)
    {
        $venue = Venue::findOrFail($id);

        $venue->update([
            'status' => 'active'
        ]);

        // Create notification
        \App\Models\Notification::create([
            'notifiable_type' => 'App\Models\User',
            'notifiable_id' => $venue->business->owner_id,
            'type' => 'venue_approved',
            'data' => json_encode([
                'title' => 'Venue Approved',
                'message' => "Your venue '{$venue->name}' has been approved and is now active.",
                'venue_id' => $venue->id
            ])
        ]);

        // Log audit
        \App\Models\AuditLog::logAction(
            'approve_venue',
            $venue,
            null,
            ['status' => 'approved']
        );

        return response()->json([
            'success' => true,
            'message' => 'Venue approved successfully'
        ]);
    }

    /**
     * Reject venue
     */
    public function reject(Request $request, $id)
    {
        $request->validate([
            'reason' => 'required|string'
        ]);

        $venue = Venue::findOrFail($id);

        $venue->update([
            'status' => 'rejected',
            'rejection_reason' => $request->reason
        ]);

        // Create notification
        \App\Models\Notification::create([
            'notifiable_type' => 'App\Models\User',
            'notifiable_id' => $venue->business->owner_id,
            'type' => 'venue_rejected',
            'data' => json_encode([
                'title' => 'Venue Rejected',
                'message' => "Your venue '{$venue->name}' has been rejected. Reason: {$request->reason}",
                'venue_id' => $venue->id
            ])
        ]);

        // Log audit
        \App\Models\AuditLog::logAction(
            'reject_venue',
            $venue,
            null,
            ['status' => 'rejected', 'reason' => $request->reason]
        );

        return response()->json([
            'success' => true,
            'message' => 'Venue rejected successfully'
        ]);
    }

    /**
     * Toggle venue status
     */
    public function toggleStatus($id)
    {
        $venue = Venue::findOrFail($id);

        $newStatus = $venue->status === 'active' ? 'inactive' : 'active';

        $venue->update([
            'status' => $newStatus
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Venue status updated',
            'status' => $venue->status
        ]);
    }

    /**
     * Toggle featured status
     */
    public function toggleFeatured($id)
    {
        $venue = Venue::findOrFail($id);

        $venue->update([
            'is_featured' => !$venue->is_featured
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Featured status updated',
            'is_featured' => $venue->is_featured
        ]);
    }

    /**
     * Update venue status
     */
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:active,inactive,under_maintenance'
        ]);

        $venue = Venue::findOrFail($id);

        $venue->update([
            'status' => $request->status
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Venue status updated',
            'status' => $venue->status
        ]);
    }

    /**
     * Delete venue
     */
    public function destroy($id)
    {
        $venue = Venue::findOrFail($id);

        // Check if venue has active offers
        if ($venue->offers()->where('status', 'active')->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete venue with active offers'
            ], 400);
        }

        $venue->delete();

        return response()->json([
            'success' => true,
            'message' => 'Venue deleted successfully'
        ]);
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:activate,deactivate,feature,unfeature,delete',
            'venue_ids' => 'required|array',
            'venue_ids.*' => 'exists:venues,id'
        ]);

        $venues = Venue::whereIn('id', $request->venue_ids);

        switch ($request->action) {
            case 'activate':
                $venues->update(['status' => 'active']);
                $message = 'Venues activated successfully';
                break;
            case 'deactivate':
                $venues->update(['status' => 'inactive']);
                $message = 'Venues deactivated successfully';
                break;
            case 'feature':
                $venues->update(['is_featured' => true]);
                $message = 'Venues featured successfully';
                break;
            case 'unfeature':
                $venues->update(['is_featured' => false]);
                $message = 'Venues unfeatured successfully';
                break;
            case 'delete':
                $venues->delete();
                $message = 'Venues deleted successfully';
                break;
        }

        return response()->json([
            'success' => true,
            'message' => $message
        ]);
    }

    /**
     * Export venues
     */
    public function export()
    {
        $venues = Venue::with(['business', 'category'])->get();

        $csv = "ID,Name,Business,Category,City,Status,Featured,Rating,Created At\n";

        foreach ($venues as $venue) {
            $csv .= sprintf(
                "%d,%s,%s,%s,%s,%s,%s,%.2f,%s\n",
                $venue->id,
                $venue->name,
                $venue->business->business_name,
                $venue->category->name,
                $venue->city,
                $venue->status,
                $venue->is_featured ? 'Yes' : 'No',
                $venue->rating,
                $venue->created_at->format('Y-m-d')
            );
        }

        return response($csv)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="venues.csv"');
    }
    /**
     * Show edit form
     */
    public function edit($id)
    {
        $venue = Venue::with(['business', 'category'])->findOrFail($id);
        $categories = Category::where('is_active', true)->get();
        $businesses = Business::where('status', 'verified')->get();

        return view('admin.venues.edit', compact('venue', 'categories', 'businesses'));
    }

    /**
     * Update venue
     */
    public function update(Request $request, $id)
    {
        $venue = Venue::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'business_id' => 'required|exists:businesses,id',
            'category_id' => 'required|exists:categories,id',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'address' => 'required|string',
            'city' => 'required|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'required|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'website' => 'nullable|url|max:255',
            'status' => 'required|in:active,inactive,pending',
            'is_featured' => 'boolean',
        ]);

        $venue->update($validated);

        // Log audit
        \App\Models\AuditLog::logAction(
            'update_venue',
            $venue,
            null,
            [
                'admin_id' => auth()->id(),
                'changes' => $validated
            ]
        );

        return redirect()->route('admin.venues.show', $venue->id)
            ->with('success', 'Venue updated successfully!');
    }
}

