<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Models\MarketingCampaign;
use App\Models\MarketingCampaignAnalytic;
use App\Models\MarketingPackage;
use App\Models\Offer;
use App\Models\Payment;
use App\Models\Venue;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class MarketingController extends Controller
{
    /**
     * Show marketing packages
     */
    public function index()
    {
        $businessId = auth()->user()->businesses()->first()->id;

        $packages = MarketingPackage::where('is_active', true)
            ->orderBy('sort_order')
            ->get()
            ->groupBy('type');

        $activeCampaigns = MarketingCampaign::where('business_id', $businessId)
            ->whereIn('status', ['active', 'approved'])
            ->count();

        $totalSpent = Payment::where('business_id', $businessId)
            ->where('type', 'marketing')
            ->where('status', 'completed')
            ->sum('amount');

        return view('business.marketing.index', compact('packages', 'activeCampaigns', 'totalSpent'));
    }

    /**
     * Show package details
     */
    public function showPackage($id)
    {
        $package = MarketingPackage::findOrFail($id);

        return view('business.marketing.package', compact('package'));
    }

    /**
     * Create campaign form
     */
    public function createCampaign($packageId)
    {
        $businessId = auth()->user()->businesses()->first()->id;

        $package = MarketingPackage::findOrFail($packageId);

        $venues = Venue::where('business_id', $businessId)
            ->where('status', 'active')
            ->get();

        $offers = Offer::whereHas('venue', function ($query) use ($businessId) {
            $query->where('business_id', $businessId);
        })->where('status', 'active')->get();

        return view('business.marketing.create-campaign', compact('package', 'venues', 'offers'));
    }

    /**
     * Store new campaign
     */
    public function storeCampaign(Request $request)
    {
        $user = auth()->user();
        $business = $user->businesses()->first();
        $businessId = $business->id;

        // Check if business has active subscription
        $activeSubscription = \App\Models\Subscription::where('business_id', $businessId)
            ->where('status', 'active')
            ->where('end_date', '>=', now())
            ->first();

        if (!$activeSubscription) {
            return redirect()
                ->route('business.subscription.plans')
                ->with('error', 'You need an active subscription to create marketing campaigns.');
        }

        $validated = $request->validate([
            'package_id' => 'required|exists:marketing_packages,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'venue_id' => 'nullable|exists:venues,id',
            'offer_id' => 'nullable|exists:offers,id',
            'media' => 'nullable|array',
            'media.*' => 'nullable|url',
            'cta_text' => 'nullable|string|max:100',
            'cta_link' => 'nullable|url',
            'start_date' => 'required|date|after_or_equal:today',
        ]);

        $package = MarketingPackage::findOrFail($validated['package_id']);

        // Calculate end date
        $startDate = \Carbon\Carbon::parse($validated['start_date']);
        $endDate = $startDate->copy()->addDays($package->duration_days);

        $campaign = MarketingCampaign::create([
            'business_id' => $businessId,
            'subscription_id' => $activeSubscription->id,
            'package_id' => $package->id,
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'type' => $package->type ?? 'mixed',
            'budget' => $package->price,
            'currency' => 'AED',
            'start_date' => $startDate,
            'end_date' => $endDate,
            'status' => 'pending',
            'creative_assets' => $validated['media'] ?? null,
            'targeting_options' => [
                'venue_id' => $validated['venue_id'] ?? null,
                'offer_id' => $validated['offer_id'] ?? null,
            ],
            'goals' => [
                'impressions' => $package->impressions ?? 0,
                'clicks' => $package->clicks ?? 0,
                'conversions' => $package->conversions ?? 0,
            ],
        ]);

        // Create payment record
        Payment::create([
            'business_id' => $businessId,
            'amount' => $package->price,
            'currency' => 'AED',
            'type' => 'marketing',
            'status' => 'pending',
            'reference_id' => $campaign->id,
            'reference_type' => 'marketing_campaign',
        ]);

        return redirect()
            ->route('business.marketing.campaigns')
            ->with('success', 'Campaign created successfully. Proceed to payment.');
    }

    /**
     * My campaigns
     */
    public function myCampaigns(Request $request)
    {
        $businessId = auth()->user()->businesses()->first()->id;

        $query = MarketingCampaign::where('business_id', $businessId)
            ->with(['package', 'venue', 'offer']);

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->type) {
            $query->where('type', $request->type);
        }

        $campaigns = $query->orderBy('created_at', 'desc')->paginate(20);

        $stats = [
            'total' => MarketingCampaign::where('business_id', $businessId)->count(),
            'active' => MarketingCampaign::where('business_id', $businessId)->where('status', 'active')->count(),
            'pending' => MarketingCampaign::where('business_id', $businessId)->where('status', 'pending')->count(),
            'completed' => MarketingCampaign::where('business_id', $businessId)->where('status', 'completed')->count(),
        ];

        return view('business.marketing.campaigns', compact('campaigns', 'stats'));
    }

    /**
     * View campaign details
     */
    public function viewCampaign($id)
    {
        $businessId = auth()->user()->businesses()->first()->id;

        $campaign = MarketingCampaign::where('business_id', $businessId)
            ->with(['package', 'venue', 'offer', 'analytics'])
            ->findOrFail($id);

        // Get analytics for last 30 days
        $analytics = MarketingCampaignAnalytic::where('campaign_id', $campaign->id)
            ->where('date', '>=', now()->subDays(30))
            ->orderBy('date')
            ->get();

        $aggregatedStats = MarketingCampaignAnalytic::getAggregatedStats($campaign->id);

        return view('business.marketing.view-campaign', compact('campaign', 'analytics', 'aggregatedStats'));
    }

    /**
     * Get campaign analytics
     */
    public function campaignAnalytics($id, Request $request)
    {
        $businessId = auth()->user()->businesses()->first()->id;

        $campaign = MarketingCampaign::where('business_id', $businessId)
            ->findOrFail($id);

        $startDate = $request->start_date ?? now()->subDays(30);
        $endDate = $request->end_date ?? now();

        $analytics = MarketingCampaignAnalytic::getForDateRange($campaign->id, $startDate, $endDate);

        $stats = [
            'total_impressions' => $analytics->sum('impressions'),
            'total_clicks' => $analytics->sum('clicks'),
            'total_conversions' => $analytics->sum('conversions'),
            'average_ctr' => $analytics->avg('ctr'),
            'cost_per_click' => $analytics->sum('clicks') > 0
                ? round($campaign->amount_paid / $analytics->sum('clicks'), 2)
                : 0,
            'cost_per_impression' => $analytics->sum('impressions') > 0
                ? round(($campaign->amount_paid / $analytics->sum('impressions')) * 1000, 2)
                : 0,
        ];

        return response()->json([
            'success' => true,
            'data' => [
                'analytics' => $analytics,
                'stats' => $stats,
            ],
        ]);
    }

    /**
     * Pause campaign
     */
    public function pauseCampaign($id)
    {
        $businessId = auth()->user()->businesses()->first()->id;

        $campaign = MarketingCampaign::where('business_id', $businessId)
            ->where('status', 'active')
            ->findOrFail($id);

        $campaign->update(['status' => 'paused']);

        return response()->json([
            'success' => true,
            'message' => 'Campaign paused successfully',
        ]);
    }

    /**
     * Resume campaign
     */
    public function resumeCampaign($id)
    {
        $businessId = auth()->user()->businesses()->first()->id;

        $campaign = MarketingCampaign::where('business_id', $businessId)
            ->where('status', 'paused')
            ->findOrFail($id);

        $campaign->update(['status' => 'active']);

        return response()->json([
            'success' => true,
            'message' => 'Campaign resumed successfully',
        ]);
    }

    /**
     * Delete draft campaign
     */
    public function deleteCampaign($id)
    {
        $businessId = auth()->user()->businesses()->first()->id;

        $campaign = MarketingCampaign::where('business_id', $businessId)
            ->where('status', 'pending')
            ->findOrFail($id);

        $campaign->delete();

        return response()->json([
            'success' => true,
            'message' => 'Campaign deleted successfully',
        ]);
    }

    /**
     * Export campaign report
     */
    public function exportReport($id)
    {
        $businessId = auth()->user()->businesses()->first()->id;

        $campaign = MarketingCampaign::where('business_id', $businessId)
            ->with(['package', 'analytics'])
            ->findOrFail($id);

        $analytics = $campaign->analytics;
        $aggregatedStats = MarketingCampaignAnalytic::getAggregatedStats($campaign->id);

        // Generate CSV
        $filename = "campaign_report_{$campaign->id}_" . date('Y-m-d') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function () use ($campaign, $analytics, $aggregatedStats) {
            $file = fopen('php://output', 'w');

            // Campaign info
            fputcsv($file, ['Campaign Report']);
            fputcsv($file, ['Campaign Name', $campaign->name]);
            fputcsv($file, ['Package', $campaign->package->name]);
            fputcsv($file, ['Start Date', $campaign->start_date->format('Y-m-d')]);
            fputcsv($file, ['End Date', $campaign->end_date->format('Y-m-d')]);
            fputcsv($file, ['Amount Paid', 'AED ' . $campaign->amount_paid]);
            fputcsv($file, []);

            // Aggregated stats
            fputcsv($file, ['Overall Statistics']);
            fputcsv($file, ['Total Impressions', $aggregatedStats['total_impressions']]);
            fputcsv($file, ['Total Clicks', $aggregatedStats['total_clicks']]);
            fputcsv($file, ['Total Conversions', $aggregatedStats['total_conversions']]);
            fputcsv($file, ['Average CTR', $aggregatedStats['average_ctr'] . '%']);
            fputcsv($file, []);

            // Daily analytics
            fputcsv($file, ['Daily Analytics']);
            fputcsv($file, ['Date', 'Impressions', 'Clicks', 'Conversions', 'CTR']);

            foreach ($analytics as $analytic) {
                fputcsv($file, [
                    $analytic->date->format('Y-m-d'),
                    $analytic->impressions,
                    $analytic->clicks,
                    $analytic->conversions,
                    $analytic->ctr . '%',
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
