<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Models\Offer;
use App\Models\Venue;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class OfferController extends Controller
{
    /**
     * Display offer listing
     */
    public function index()
    {
        $business = auth()->user()->business;

        if (!$business) {
            return redirect()->route('business.register')
                ->with('error', 'Please complete business registration first.');
        }

        $offers = Offer::whereHas('venue', function($query) use ($business) {
            $query->where('business_id', $business->id);
        })
            ->with(['venue', 'category'])
            ->withCount('redemptions')
            ->orderBy('created_at', 'desc')
            ->get();

        $stats = [
            'total' => $offers->count(),
            'active' => $offers->where('status', 'active')->count(),
            'expired' => $offers->where('status', 'expired')->count(),
            'total_redemptions' => $offers->sum('redemption_count'),
        ];

        $venues = $business->venues()->where('status', 'active')->get();

        return view('business.offers.index', compact('offers', 'stats', 'venues'));
    }

    /**
     * Show offer details
     */
    public function show($id)
    {
        $business = auth()->user()->business;

        $offer = Offer::whereHas('venue', function($query) use ($business) {
            $query->where('business_id', $business->id);
        })
            ->with(['venue', 'category', 'redemptions.user'])
            ->findOrFail($id);

        $stats = [
            'total_redemptions' => $offer->redemption_count,
            'views' => $offer->view_count ?? 0,
            'conversion_rate' => $offer->view_count > 0 ? round(($offer->redemption_count / $offer->view_count) * 100, 2) : 0,
        ];

        return view('business.offers.show', compact('offer', 'stats'));
    }

    /**
     * Show create form
     */
    public function create()
    {
        $business = auth()->user()->business;

        if (!$business->canCreateOffers()) {
            $message = 'You need an active subscription and approved payment to create offers.';
            if (!$business->isVerified()) {
                $message = 'Your business must be verified first.';
            } elseif (!$business->hasApprovedPayment()) {
                $message = 'Please complete payment before creating offers.';
            } elseif (!$business->hasActiveSubscription()) {
                $message = 'You need an active subscription to create offers.';
            }
            return redirect()->route('business.subscription.index')
                ->with('error', $message);
        }

        $venues = $business->venues()->where('status', 'active')->get();

        if ($venues->isEmpty()) {
            return redirect()->route('business.venues.index')
                ->with('error', 'Please create a venue first before creating offers.');
        }

        return view('business.offers.create', compact('venues'));
    }

    /**
     * Store new offer
     */
    public function store(Request $request)
    {
        $business = auth()->user()->business;

        // Verify business can create offers
        if (!$business->canCreateOffers()) {
            return back()->withErrors([
                'permission' => 'You need an active subscription and approved payment to create offers.'
            ])->withInput();
        }

        // Check subscription limits
        $plan = $business->currentSubscription?->plan;
        if ($plan && $plan->max_offers) {
            $offerCount = Offer::whereHas('venue', function($query) use ($business) {
                $query->where('business_id', $business->id);
            })->where('status', 'active')->count();

            if ($offerCount >= $plan->max_offers) {
                return back()->withErrors([
                    'limit' => 'You have reached your active offer limit. Please upgrade your plan.'
                ])->withInput();
            }
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'title_ar' => 'nullable|string|max:255',
            'description' => 'required|string',
            'description_ar' => 'nullable|string',
            'venue_ids' => 'required|array|min:1',
            'venue_ids.*' => 'exists:venues,id',
            'offer_type' => 'required|in:regular,special,exclusive',
            'discount_type' => 'required|in:percentage,fixed,bogo,free_item',
            'discount_value' => 'required|numeric|min:0',
            'original_price' => 'nullable|numeric|min:0',
            'discounted_price' => 'nullable|numeric|min:0',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'terms_and_conditions' => 'nullable|string',
            'terms_and_conditions_ar' => 'nullable|string',
            'max_redemptions' => 'nullable|integer|min:1',
            'max_redemptions_per_user' => 'nullable|integer|min:1',
            'image' => 'nullable|image|mimes:jpeg,png,jpg|max:5120',
        ]);

        // Verify venues belong to business
        $venues = Venue::where('business_id', $business->id)
            ->whereIn('id', $validated['venue_ids'])
            ->get();

        if ($venues->count() !== count($validated['venue_ids'])) {
            return back()->withErrors(['venue_ids' => 'Invalid venue selection'])->withInput();
        }

        $validated['slug'] = Str::slug($validated['title']) . '-' . Str::random(6);

        // Handle image upload
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('offers', 'public');
            $validated['image'] = $path;
        }
        $validated['business_id'] = $business->id;
        // Create offer for each venue
        $venueIds = $validated['venue_ids'];
        unset($validated['venue_ids']);

        foreach ($venueIds as $venueId) {
            $offerData = $validated;
            $offerData['venue_id'] = $venueId;
            $offerData['category_id'] = Venue::find($venueId)->category_id;

            Offer::create($offerData);
        }

        return redirect()->route('business.offers.index')
            ->with('success', 'Offer created successfully for ' . count($venueIds) . ' venue(s)!');
    }

    /**
     * Show edit form
     */
    public function edit($id)
    {
        $business = auth()->user()->business;

        $offer = Offer::whereHas('venue', function($query) use ($business) {
            $query->where('business_id', $business->id);
        })->findOrFail($id);

        $venues = $business->venues()->where('status', 'active')->get();

        return view('business.offers.edit', compact('offer', 'venues'));
    }

    /**
     * Update offer
     */
    public function update(Request $request, $id)
    {
        $business = auth()->user()->business;

        $offer = Offer::whereHas('venue', function($query) use ($business) {
            $query->where('business_id', $business->id);
        })->findOrFail($id);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'title_ar' => 'nullable|string|max:255',
            'description' => 'required|string',
            'description_ar' => 'nullable|string',
            'venue_id' => 'required|exists:venues,id',
            'offer_type' => 'required|in:regular,special,exclusive',
            'discount_type' => 'required|in:percentage,fixed,bogo,free_item',
            'discount_value' => 'required|numeric|min:0',
            'original_price' => 'nullable|numeric|min:0',
            'discounted_price' => 'nullable|numeric|min:0',
            'valid_from' => 'required|date',
            'valid_until' => 'required|date|after:valid_from',
            'terms_and_conditions' => 'nullable|string',
            'terms_and_conditions_ar' => 'nullable|string',
            'max_redemptions' => 'nullable|integer|min:1',
            'max_redemptions_per_user' => 'nullable|integer|min:1',
            'status' => 'required|in:active,inactive,expired',
            'image' => 'nullable|image|mimes:jpeg,png,jpg|max:5120',
        ]);

        // Verify venue belongs to business
        $venue = Venue::where('business_id', $business->id)
            ->findOrFail($validated['venue_id']);

        $validated['category_id'] = $venue->category_id;

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image
            if ($offer->image) {
                Storage::disk('public')->delete($offer->image);
            }
            $path = $request->file('image')->store('offers', 'public');
            $validated['image'] = $path;
        }

        $offer->update($validated);

        return redirect()->route('business.offers.index')
            ->with('success', 'Offer updated successfully!');
    }

    /**
     * Delete offer
     */
    public function destroy($id)
    {
        $business = auth()->user()->business;

        $offer = Offer::whereHas('venue', function($query) use ($business) {
            $query->where('business_id', $business->id);
        })->findOrFail($id);

        // Delete image
        if ($offer->image) {
            Storage::disk('public')->delete($offer->image);
        }

        $offer->delete();

        return response()->json([
            'success' => true,
            'message' => 'Offer deleted successfully'
        ]);
    }

    /**
     * Toggle offer status
     */
    public function toggleStatus($id)
    {
        $business = auth()->user()->business;

        $offer = Offer::whereHas('venue', function($query) use ($business) {
            $query->where('business_id', $business->id);
        })->findOrFail($id);

        $newStatus = $offer->status === 'active' ? 'inactive' : 'active';
        $offer->update(['status' => $newStatus]);

        return response()->json([
            'success' => true,
            'message' => 'Offer status updated',
            'status' => $newStatus
        ]);
    }

    /**
     * Duplicate offer
     */
    public function duplicate($id)
    {
        $business = auth()->user()->business;

        $offer = Offer::whereHas('venue', function($query) use ($business) {
            $query->where('business_id', $business->id);
        })->findOrFail($id);

        $newOffer = $offer->replicate();
        $newOffer->title = $offer->title . ' (Copy)';
        $newOffer->slug = Str::slug($newOffer->title) . '-' . Str::random(6);
        $newOffer->status = 'inactive';
        $newOffer->redemption_count = 0;
        $newOffer->save();

        return response()->json([
            'success' => true,
            'message' => 'Offer duplicated successfully',
            'offer_id' => $newOffer->id
        ]);
    }
}
