<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class SettingsController extends Controller
{
    /**
     * Display settings page
     */
    public function index()
    {
        $business = auth()->user()->business;
        $user = auth()->user();

        return view('business.settings.index', compact('business', 'user'));
    }

    /**
     * Update business settings
     */
    public function updateAccount(Request $request)
    {
        $business = auth()->user()->business;

        $validated = $request->validate([
            'business_name' => 'required|string|max:255',
            'business_name_ar' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'website' => 'nullable|url|max:255',
            'address' => 'nullable|string',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        if ($request->hasFile('logo')) {
            if ($business->logo) {
                Storage::disk('public')->delete($business->logo);
            }
            $validated['logo'] = $request->file('logo')->store('businesses/logos', 'public');
        }
//        change the status to pending on update
        $validated['status'] = 'pending';
        $business->update($validated);

        return back()->with('success', 'Settings updated successfully!');
    }

    /**
     * Update profile
     */
    public function updateProfile(Request $request)
    {
        $user = auth()->user();

        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'required|string|max:20',
        ]);

        $user->update($validated);

        return back()->with('success', 'Profile updated successfully!');
    }

    /**
     * Update password
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'password' => 'required|string|min:8|confirmed',
        ]);

        $user = auth()->user();

        if (!Hash::check($request->current_password, $user->password)) {
            return back()->withErrors(['current_password' => 'Current password is incorrect']);
        }

        $user->update([
            'password' => Hash::make($request->password)
        ]);

        return back()->with('success', 'Password updated successfully!');
    }

    /**
     * Show notification preferences page
     */
    public function notificationPreferences()
    {
        $user = auth()->user();
        $preferences = json_decode($user->notification_preferences ?? '{}', true);

        return view('business.settings.notifications', compact('preferences'));
    }

    /**
     * Update notification preferences
     */
    public function updateNotifications(Request $request)
    {
        $user = auth()->user();

        $preferences = [
            'email_notifications' => $request->boolean('email_notifications'),
            'sms_notifications' => $request->boolean('sms_notifications'),
            'push_notifications' => $request->boolean('push_notifications'),
            'marketing_emails' => $request->boolean('marketing_emails'),
        ];

        $user->update([
            'notification_preferences' => json_encode($preferences)
        ]);

        return back()->with('success', 'Notification preferences updated!');
    }
}
