<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Payment\StripeController;
use App\Models\SubscriptionPlan;
use App\Models\Subscription;
use App\Models\Payment;
use Illuminate\Http\Request;

class SubscriptionController extends Controller
{
    /**
     * Display current subscription
     */
    public function index()
    {
        $business = auth()->user()->business;
        $subscription = $business->subscriptions()
            ->with('plan')
            ->where('status', 'active')
            ->first();

        $usage = [
            'venues' => $business->venues()->count(),
            'offers' => $business->venues()->withCount('offers')->get()->sum('offers_count'),
            'staff' => $business->staff()->count(),
        ];

        $invoices = Payment::where('business_id', $business->id)
            ->whereNotNull('subscription_id')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('business.subscription.index', compact('subscription', 'usage', 'invoices'));
    }

    /**
     * Show available plans
     */
    public function plans()
    {
        $business = auth()->user()->business;
        $currentSubscription = $business->subscriptions()
            ->where('status', 'active')
            ->first();

        $plans = SubscriptionPlan::where('is_active', true)
            ->where('type', 'business')
            ->orderBy('price')
            ->get();

        return view('business.subscription.plans', compact('plans', 'currentSubscription'));
    }

    /**
     * Upgrade/Change plan
     */
    public function upgrade(Request $request)
    {
        $request->validate([
            'plan_id' => 'required|exists:subscription_plans,id',
            'payment_method' => 'nullable|in:stripe,paypal'
        ]);

        $business = auth()->user()->business;

        if (!$business) {
            return redirect()->back()->with('error', 'Business not found. Please complete your business profile first.');
        }

        $newPlan = SubscriptionPlan::findOrFail($request->plan_id);

        // Ensure plan type is business
        if ($newPlan->type !== 'business') {
            return redirect()->back()->with('error', 'Selected plan is not available for businesses.');
        }

        // Check if plan is free (price is 0)
        $isFree = $newPlan->price == 0;

        // Validate payment method only for paid plans
        if (!$isFree && !$request->payment_method) {
            return redirect()->back()->with('error', 'Please select a payment method.');
        }

        // Check current subscription
        $currentSubscription = $business->subscriptions()
            ->where('status', 'active')
            ->first();

        if ($currentSubscription) {
            // Cancel current subscription
            $currentSubscription->update([
                'status' => 'cancelled',
                'cancelled_at' => now()
            ]);
        }

        // Calculate end date based on duration
        $endDate = match($newPlan->duration) {
            'monthly' => now()->addMonth(),
            'quarterly' => now()->addMonths(3),
            'yearly' => now()->addYear(),
            default => now()->addMonth(),
        };

        // Create new subscription
        $subscription = Subscription::create([
            'user_id' => auth()->id(),
            'business_id' => $business->id,
            'subscription_plan_id' => $newPlan->id,
            'status' => $isFree ? 'active' : 'pending',
            'start_date' => now(),
            'end_date' => $endDate,
            'auto_renew' => true
        ]);

        // Create payment record
        $payment = Payment::create([
            'user_id' => auth()->id(),
            'business_id' => $business->id,
            'subscription_id' => $subscription->id,
            'amount' => $newPlan->price,
            'currency' => $newPlan->currency,
            'payment_method' => $isFree ? 'free' : $request->payment_method,
            'status' => $isFree ? 'completed' : 'pending',
            'type' => 'subscription',
            'payment_date' => $isFree ? now() : null
        ]);

        // If plan is free, activate subscription immediately
        if ($isFree) {
            // Update business profile
            $business->update([
                'subscription_plan_id' => $subscription->subscription_plan_id,
                'subscription_status' => 'active',
                'subscription_start_date' => $subscription->start_date,
                'subscription_end_date' => $subscription->end_date,
            ]);

            return redirect()->route('business.subscription.index')
                ->with('success', 'Free plan activated successfully!');
        }

        return redirect()->route('business.subscription.checkout', $subscription->id)
            ->with('success', 'Plan selected! Please complete payment.');
    }

    /**
     * Show checkout page
     */
    public function checkout($subscriptionId)
    {
        $subscription = Subscription::with('plan')
            ->where('business_id', auth()->user()->business->id)
            ->findOrFail($subscriptionId);

        $payment = Payment::where('subscription_id', $subscription->id)
            ->where('status', 'pending')
            ->firstOrFail();

        return view('business.subscription.checkout', compact('subscription', 'payment'));
    }

    /**
     * Show invoice
     */
    public function showInvoice($id)
    {
        $business = auth()->user()->business;
        $invoice = Payment::where('business_id', $business->id)
            ->findOrFail($id);

        return view('business.subscription.invoice', compact('invoice'));
    }

    /**
     * Download invoice
     */
    public function downloadInvoice($id)
    {
        $business = auth()->user()->business;
        $invoice = Payment::where('business_id', $business->id)
            ->findOrFail($id);

        // Generate PDF (simplified version)
        $html = view('business.subscription.invoice-pdf', compact('invoice'))->render();

        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="invoice-' . $invoice->id . '.html"');
    }

    /**
     * Cancel subscription
     */
    public function cancel(Request $request)
    {
        $request->validate([
            'reason' => 'nullable|string|max:500'
        ]);

        $business = auth()->user()->business;
        $subscription = $business->subscriptions()
            ->where('status', 'active')
            ->firstOrFail();

        $subscription->update([
            'status' => 'cancelled',
            'cancelled_at' => now(),
            'cancellation_reason' => $request->reason,
            'auto_renew' => false
        ]);

        return redirect()->route('business.subscription.index')
            ->with('success', 'Subscription cancelled successfully.');
    }

    /**
     * Reactivate subscription
     */
    public function reactivate()
    {
        $business = auth()->user()->business;
        $subscription = $business->subscriptions()
            ->where('status', 'cancelled')
            ->where('end_date', '>', now())
            ->firstOrFail();

        $subscription->update([
            'status' => 'active',
            'cancelled_at' => null,
            'auto_renew' => true
        ]);

        return redirect()->route('business.subscription.index')
            ->with('success', 'Subscription reactivated successfully.');
    }

    /**
     * View invoices
     */
    public function invoices()
    {
        $business = auth()->user()->business;

        $invoices = Payment::where('business_id', $business->id)
            ->whereNotNull('subscription_id')
            ->with('subscription.plan')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('business.subscription.invoices', compact('invoices'));
    }

    /**
     * Download invoice
     */
    public function updatePayment(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|exists:payments,id',
            'payment_method' => 'required|in:stripe,paypal'
        ]);

        $business = auth()->user()->business;
        $payment = Payment::where('business_id', $business->id)
            ->where('status', 'pending')
            ->findOrFail($request->payment_id);

        $payment->update([
            'payment_method' => $request->payment_method
        ]);

        return redirect()->route('business.subscription.checkout', $payment->subscription_id)
            ->with('success', 'Payment method updated! Please complete payment.');
    }
}
