<?php

namespace App\Http\Controllers\Payment;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Services\Payment\StripeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class StripeController extends Controller
{
    protected $stripeService;

    public function __construct(StripeService $stripeService)
    {
        $this->stripeService = $stripeService;
    }

    /**
     * Create payment intent
     */
    public function createIntent(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|exists:payments,id',
        ]);

        $payment = Payment::findOrFail($request->payment_id);

        if ($payment->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Payment is not in pending status'
            ], 400);
        }

        try {
            $intent = $this->stripeService->createPaymentIntent(
                $payment->amount,
                $payment->currency,
                [
                    'payment_id' => $payment->id,
                    'user_id' => $payment->user_id,
                    'type' => $payment->type
                ]
            );

            $payment->update([
                'transaction_id' => $intent->id
            ]);

            return response()->json([
                'success' => true,
                'client_secret' => $intent->client_secret,
                'payment_intent_id' => $intent->id
            ]);
        } catch (\Exception $e) {
            Log::error('Stripe payment intent creation failed: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to create payment intent'
            ], 500);
        }
    }

    /**
     * Process payment
     */
    public function process(Request $request)
    {
        $request->validate([
            'payment_intent_id' => 'required|string',
            'payment_id' => 'required|exists:payments,id'
        ]);

        $payment = Payment::findOrFail($request->payment_id);

        try {
            $intent = $this->stripeService->retrievePaymentIntent($request->payment_intent_id);

            if ($intent->status === 'succeeded') {
                $payment->update([
                    'status' => 'completed',
                    'transaction_id' => $intent->id,
                    'payment_details' => json_encode([
                        'stripe_payment_intent' => $intent->id,
                        'amount_received' => $intent->amount_received / 100,
                        'payment_method' => $intent->payment_method,
                    ])
                ]);

                // Activate subscription if payment is for subscription
                if ($payment->subscription_id) {
                    $subscription = $payment->subscription;
                    $subscription->update(['status' => 'active']);

                    // Update business profile if this is a business subscription
                    if ($payment->business_id && $subscription->plan) {
                        $business = \App\Models\Business::find($payment->business_id);

                        if ($business) {
                            $business->update([
                                'subscription_plan_id' => $subscription->subscription_plan_id,
                                'subscription_status' => 'active',
                                'subscription_start_date' => $subscription->start_date,
                                'subscription_end_date' => $subscription->end_date,
                            ]);

                            Log::info('Business subscription updated', [
                                'business_id' => $business->id,
                                'plan_id' => $subscription->subscription_plan_id,
                                'subscription_id' => $subscription->id
                            ]);
                        }
                    }
                }

                return response()->json([
                    'success' => true,
                    'message' => 'Payment processed successfully',
                    'payment' => $payment
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'Payment not successful',
                'status' => $intent->status
            ], 400);

        } catch (\Exception $e) {
            Log::error('Stripe payment processing failed: ' . $e->getMessage());

            $payment->update(['status' => 'failed']);

            return response()->json([
                'success' => false,
                'message' => 'Payment processing failed'
            ], 500);
        }
    }

    /**
     * Handle Stripe webhook
     */
    public function webhook(Request $request)
    {
        $payload = $request->getContent();
        $sig_header = $request->header('Stripe-Signature');
        $endpoint_secret = config('services.stripe.webhook_secret');

        try {
            $event = \Stripe\Webhook::constructEvent(
                $payload,
                $sig_header,
                $endpoint_secret
            );
        } catch (\Exception $e) {
            Log::error('Stripe webhook error: ' . $e->getMessage());
            return response()->json(['error' => 'Webhook error'], 400);
        }

        // Handle the event
        switch ($event->type) {
            case 'payment_intent.succeeded':
                $paymentIntent = $event->data->object;
                $this->handlePaymentSuccess($paymentIntent);
                break;

            case 'payment_intent.payment_failed':
                $paymentIntent = $event->data->object;
                $this->handlePaymentFailure($paymentIntent);
                break;

            case 'charge.refunded':
                $charge = $event->data->object;
                $this->handleRefund($charge);
                break;

            default:
                Log::info('Unhandled Stripe webhook event: ' . $event->type);
        }

        return response()->json(['success' => true]);
    }

    /**
     * Handle successful payment
     */
    protected function handlePaymentSuccess($paymentIntent)
    {
        $payment = Payment::where('transaction_id', $paymentIntent->id)->first();

        if ($payment && $payment->status === 'pending') {
            $payment->update([
                'status' => 'completed',
                'payment_details' => json_encode($paymentIntent)
            ]);

            if ($payment->subscription_id) {
                $payment->subscription->update(['status' => 'active']);
            }
        }
    }

    /**
     * Handle failed payment
     */
    protected function handlePaymentFailure($paymentIntent)
    {
        $payment = Payment::where('transaction_id', $paymentIntent->id)->first();

        if ($payment) {
            $payment->update([
                'status' => 'failed',
                'payment_details' => json_encode($paymentIntent)
            ]);
        }
    }

    /**
     * Handle refund
     */
    protected function handleRefund($charge)
    {
        $payment = Payment::where('transaction_id', $charge->payment_intent)->first();

        if ($payment) {
            $payment->update([
                'status' => 'refunded',
                'payment_details' => json_encode($charge)
            ]);
        }
    }

    /**
     * Process refund
     */
    public function refund(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|exists:payments,id',
            'amount' => 'nullable|numeric|min:0',
            'reason' => 'nullable|string'
        ]);

        $payment = Payment::findOrFail($request->payment_id);

        if ($payment->status !== 'completed') {
            return response()->json([
                'success' => false,
                'message' => 'Only completed payments can be refunded'
            ], 400);
        }

        try {
            $refund = $this->stripeService->refundPayment(
                $payment->transaction_id,
                $request->amount ? $request->amount * 100 : null,
                $request->reason
            );

            $payment->update([
                'status' => 'refunded',
                'payment_details' => json_encode(array_merge(
                    json_decode($payment->payment_details, true) ?? [],
                    ['refund' => $refund]
                ))
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Refund processed successfully',
                'refund' => $refund
            ]);

        } catch (\Exception $e) {
            Log::error('Stripe refund failed: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Refund processing failed'
            ], 500);
        }
    }
}
