<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Business extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'owner_id',
        'business_name',
        'business_name_ar',
        'trade_license_number',
        'trade_license_document',
        'email',
        'phone',
        'website',
        'description',
        'description_ar',
        'logo',
        'cover_image',
        'status',
        'verification_notes',
        'verified_at',
        'verified_by',
        'subscription_plan_id',
        'subscription_status',
        'subscription_start_date',
        'subscription_end_date',
        'total_venues',
        'total_staff',
        'total_offers',
        'rating',
        'total_reviews',
    ];

    protected $casts = [
        'verified_at' => 'datetime',
        'subscription_start_date' => 'date',
        'subscription_end_date' => 'date',
        'rating' => 'decimal:2',
    ];

    public function owner()
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    public function verifier()
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    public function subscriptionPlan()
    {
        return $this->belongsTo(SubscriptionPlan::class);
    }

    public function venues()
    {
        return $this->hasMany(Venue::class);
    }

    public function offers()
    {
        return $this->hasMany(Offer::class);
    }

    public function subscriptions()
    {
        return $this->hasMany(Subscription::class);
    }

    public function activeSubscription()
    {
        return $this->hasOne(Subscription::class)
            ->where('status', 'active')
            ->where('end_date', '>', now())
            ->latest();
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    public function followers()
    {
        return $this->hasMany(Follow::class);
    }

    public function staff()
    {
        return $this->hasMany(BusinessStaff::class);
    }

    public function activeStaff()
    {
        return $this->hasMany(BusinessStaff::class)->where('status', 'active');
    }

    public function pendingInvitations()
    {
        return $this->hasMany(BusinessStaff::class)->where('status', 'pending_invitation');
    }

    public function managers()
    {
        return $this->hasMany(BusinessStaff::class)
            ->where('role', 'manager')
            ->where('status', 'active');
    }

    public function isVerified()
    {
        return $this->status === 'verified';
    }

    public function hasActiveSubscription()
    {
        return $this->subscription_status === 'active'
            && $this->subscription_end_date
            && $this->subscription_end_date->isFuture();
    }

    public function canAddMoreStaff()
    {
        if (!$this->subscriptionPlan || !$this->subscriptionPlan->max_staff) {
            return true; // No limit
        }

        $currentStaffCount = $this->staff()
            ->where('status', '!=', 'inactive')
            ->count();

        return $currentStaffCount < $this->subscriptionPlan->max_staff;
    }

    public function hasApprovedPayment()
    {
        return $this->payments()
            ->where('status', 'completed')
            ->exists();
    }

    public function canCreateVenues()
    {
        return $this->isVerified() && $this->hasApprovedPayment() && $this->hasActiveSubscription();
    }

    public function canCreateOffers()
    {
        return $this->isVerified() && $this->hasApprovedPayment() && $this->hasActiveSubscription();
    }
}
