<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Spotlight extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id',
        'venue_id',
        'offer_id',
        'redemption_id',
        'content',
        'media_type',
        'media_urls',
        'location',
        'latitude',
        'longitude',
        'status',
        'moderation_status',
        'moderated_by',
        'moderated_at',
        'moderation_notes',
        'flag_status',
        'flag_reason',
        'flagged_by',
        'flagged_at',
        'reviewed_by',
        'reviewed_at',
        'likes_count',
        'comments_count',
        'shares_count',
        'view_count',
    ];

    protected $casts = [
        'media_urls' => 'array',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'moderated_at' => 'datetime',
        'flagged_at' => 'datetime',
        'reviewed_at' => 'datetime',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function moderator()
    {
        return $this->belongsTo(User::class, 'moderated_by');
    }

    public function likes()
    {
        return $this->hasMany(SpotlightLike::class);
    }

    public function comments()
    {
        return $this->hasMany(SpotlightComment::class);
    }

    public function scopeApproved($query)
    {
        return $query->where('moderation_status', 'approved')
            ->where('status', 'approved');
    }

    public function scopePending($query)
    {
        return $query->where('moderation_status', 'pending');
    }

    public function scopeFlagged($query)
    {
        return $query->where('status', 'flagged');
    }

    public function isApproved()
    {
        return $this->moderation_status === 'approved';
    }

    public function isPending()
    {
        return $this->moderation_status === 'pending';
    }

    public function isLikedBy(User $user)
    {
        return $this->likes()->where('user_id', $user->id)->exists();
    }

    public function incrementLikes()
    {
        $this->increment('likes_count');
    }

    public function decrementLikes()
    {
        $this->decrement('likes_count');
    }

    public function incrementComments()
    {
        $this->increment('comments_count');
    }

    public function decrementComments()
    {
        $this->decrement('comments_count');
    }

    public function incrementViews()
    {
        $this->increment('view_count');
    }

    public function venue()
    {
        return $this->belongsTo(Venue::class);
    }

    public function offer()
    {
        return $this->belongsTo(Offer::class);
    }

    public function redemption()
    {
        return $this->belongsTo(Redemption::class);
    }

    public function flagger()
    {
        return $this->belongsTo(User::class, 'flagged_by');
    }

    public function reviewer()
    {
        return $this->belongsTo(User::class, 'reviewed_by');
    }

    public function scopeNotFlagged($query)
    {
        return $query->where('flag_status', 'none');
    }

    public function scopeUnderReview($query)
    {
        return $query->whereIn('flag_status', ['flagged', 'under_review']);
    }

    public function isFlagged()
    {
        return in_array($this->flag_status, ['flagged', 'under_review']);
    }

    public function canBeViewed()
    {
        return $this->isApproved() && !$this->isFlagged();
    }

    public function flagForReview(User $user, string $reason)
    {
        $this->update([
            'flag_status' => 'under_review',
            'flag_reason' => $reason,
            'flagged_by' => $user->id,
            'flagged_at' => now(),
        ]);
    }

    public function clearFlag(User $reviewer)
    {
        $this->update([
            'flag_status' => 'cleared',
            'reviewed_by' => $reviewer->id,
            'reviewed_at' => now(),
        ]);
    }

    /**
     * Check if user can post more spotlights for this redemption
     */
    public static function canPostForRedemption(int $redemptionId, int $userId): bool
    {
        $maxSpotlights = (int) config('app.max_spotlights_per_redemption', 3);

        $count = self::where('redemption_id', $redemptionId)
            ->where('user_id', $userId)
            ->count();

        return $count < $maxSpotlights;
    }
}
