<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class SystemConfiguration extends Model
{
    protected $fillable = [
        'key',
        'group',
        'value',
        'type',
        'description',
        'is_public',
        'options',
        'sort_order',
    ];

    protected $casts = [
        'is_public' => 'boolean',
        'options' => 'array',
        'sort_order' => 'integer',
    ];

    /**
     * Get configuration value by key
     */
    public static function get(string $key, $default = null)
    {
        return Cache::remember("config.{$key}", 3600, function () use ($key, $default) {
            $config = self::where('key', $key)->first();

            if (!$config) {
                return $default;
            }

            return self::castValue($config->value, $config->type);
        });
    }

    /**
     * Set configuration value
     */
    public static function set(string $key, $value): bool
    {
        $config = self::where('key', $key)->first();

        if (!$config) {
            return false;
        }

        $config->value = is_array($value) ? json_encode($value) : $value;
        $config->save();

        Cache::forget("config.{$key}");

        return true;
    }

    /**
     * Get all configurations by group
     */
    public static function getGroup(string $group): array
    {
        return Cache::remember("config.group.{$group}", 3600, function () use ($group) {
            return self::where('group', $group)
                ->orderBy('sort_order')
                ->get()
                ->mapWithKeys(function ($config) {
                    return [$config->key => self::castValue($config->value, $config->type)];
                })
                ->toArray();
        });
    }

    /**
     * Get all public configurations
     */
    public static function getPublic(): array
    {
        return Cache::remember('config.public', 3600, function () {
            return self::where('is_public', true)
                ->get()
                ->mapWithKeys(function ($config) {
                    return [$config->key => self::castValue($config->value, $config->type)];
                })
                ->toArray();
        });
    }

    /**
     * Cast value based on type
     */
    protected static function castValue($value, string $type)
    {
        switch ($type) {
            case 'boolean':
                return filter_var($value, FILTER_VALIDATE_BOOLEAN);
            case 'integer':
                return (int) $value;
            case 'float':
                return (float) $value;
            case 'json':
            case 'array':
                return is_string($value) ? json_decode($value, true) : $value;
            default:
                return $value;
        }
    }

    /**
     * Clear all configuration cache
     */
    public static function clearCache(): void
    {
        Cache::flush();
    }

    /**
     * Check if a feature is enabled
     */
    public static function isEnabled(string $featureKey): bool
    {
        return (bool) self::get($featureKey, false);
    }

    /**
     * Get SMS provider
     */
    public static function getSmsProvider(): string
    {
        return self::get('sms.provider', 'twilio');
    }

    /**
     * Check if social login is enabled
     */
    public static function isSocialLoginEnabled(): bool
    {
        return self::isEnabled('auth.allow_social_login');
    }

    /**
     * Check if referral is enabled
     */
    public static function isReferralEnabled(): bool
    {
        return self::isEnabled('referral.enabled');
    }

    /**
     * Check if chat is enabled
     */
    public static function isChatEnabled(): bool
    {
        return self::isEnabled('chat.enabled');
    }

    /**
     * Check if GDPR is enabled
     */
    public static function isGdprEnabled(): bool
    {
        return self::isEnabled('gdpr.enabled');
    }

    /**
     * Get enabled languages
     */
    public static function getEnabledLanguages(): array
    {
        return self::get('app.languages', ['en']);
    }

    /**
     * Get offer types
     */
    public static function getOfferTypes(): array
    {
        return self::get('offer.types', ['regular', 'special', 'exclusive']);
    }
}
