<?php

namespace App\Services;

use Twilio\Rest\Client;
use Illuminate\Support\Facades\Log;

class SMSService
{
    protected $twilio;
    protected $from;

    public function __construct()
    {
        $sid = config('services.twilio.sid');
        $token = config('services.twilio.token');
        $this->from = config('services.twilio.from');

        if ($sid && $token) {
            $this->twilio = new Client($sid, $token);
        }
    }

    /**
     * Send OTP code via SMS
     */
    public function sendOTP($phone, $code)
    {
        try {
            $message = "Your Privasee verification code is: {$code}. Valid for 10 minutes. Do not share this code.";

            if ($this->twilio) {
                $this->twilio->messages->create(
                    $phone,
                    [
                        'from' => $this->from,
                        'body' => $message
                    ]
                );

                Log::info('SMS OTP sent successfully', ['phone' => $phone]);

                return [
                    'success' => true,
                    'message' => 'OTP sent successfully'
                ];
            }

            // Fallback for development (log OTP instead of sending)
            Log::info('OTP Code (Dev Mode)', [
                'phone' => $phone,
                'code' => $code
            ]);

            return [
                'success' => true,
                'message' => 'OTP logged (dev mode)',
                'code' => config('app.debug') ? $code : null
            ];

        } catch (\Exception $e) {
            Log::error('Failed to send SMS OTP', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Failed to send OTP: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Send custom SMS message
     */
    public function send($phone, $message)
    {
        try {
            if ($this->twilio) {
                $this->twilio->messages->create(
                    $phone,
                    [
                        'from' => $this->from,
                        'body' => $message
                    ]
                );

                Log::info('SMS sent successfully', ['phone' => $phone]);

                return [
                    'success' => true,
                    'message' => 'SMS sent successfully'
                ];
            }

            Log::info('SMS (Dev Mode)', [
                'phone' => $phone,
                'message' => $message
            ]);

            return [
                'success' => true,
                'message' => 'SMS logged (dev mode)'
            ];

        } catch (\Exception $e) {
            Log::error('Failed to send SMS', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Failed to send SMS: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Validate phone number format
     */
    public function validatePhone($phone)
    {
        // Remove spaces and special characters
        $phone = preg_replace('/[^0-9+]/', '', $phone);

        // Check if it starts with +
        if (!str_starts_with($phone, '+')) {
            return false;
        }

        // Check minimum length (country code + number)
        if (strlen($phone) < 10) {
            return false;
        }

        return true;
    }

    /**
     * Format phone number to E.164 format
     */
    public function formatPhone($phone, $defaultCountryCode = '+971')
    {
        // Remove all non-numeric characters except +
        $phone = preg_replace('/[^0-9+]/', '', $phone);

        // If doesn't start with +, add default country code
        if (!str_starts_with($phone, '+')) {
            // Remove leading zeros
            $phone = ltrim($phone, '0');
            $phone = $defaultCountryCode . $phone;
        }

        return $phone;
    }

    /**
     * Generate 6-digit OTP code
     */
    public function generateOTP()
    {
        return str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
    }
}
