@extends('layouts.admin')

@section('title', 'Payments Management')

@section('content')
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h2 class="fw-bold mb-1">Payments Management</h2>
            <p class="text-muted mb-0">View all payment transactions</p>
        </div>
    </div>

    <div class="admin-card">
        <div class="admin-card-header">
            <h5 class="mb-0 fw-bold">All Payments</h5>
        </div>
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead>
                    <tr>
                        <th>Transaction ID</th>
                        <th>Business</th>
                        <th>Amount</th>
                        <th>Method</th>
                        <th>Status</th>
                        <th>Date</th>
                        <th>Actions</th>
                    </tr>
                    </thead>
                    <tbody>
                    @forelse($payments as $payment)
                        <tr>
                            <td><code>{{ $payment->transaction_id }}</code></td>
                            <td>{{ $payment->business->business_name ?? 'N/A' }}</td>
                            <td><strong>{{ $payment->currency }} {{ number_format($payment->amount, 2) }}</strong></td>
                            <td>{{ ucfirst($payment->payment_method) }}</td>
                            <td>
                            <span class="badge bg-{{ $payment->status === 'completed' ? 'success' : ($payment->status === 'pending' ? 'warning' : 'danger') }}">
                                {{ ucfirst($payment->status) }}
                            </span>
                            </td>
                            <td>{{ $payment->payment_date ? $payment->payment_date->format('M d, Y') : 'N/A' }}</td>
                            <td>
                                <button class="btn btn-sm btn-outline-primary" onclick="viewPayment({{ $payment->id }})">
                                    <i class="bi bi-eye me-1"></i>View
                                </button>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="7" class="text-center text-muted py-4">No payments found</td>
                        </tr>
                    @endforelse
                    </tbody>
                </table>
            </div>
        </div>
        @if(isset($payments) && $payments->hasPages())
            <div class="admin-card-footer">
                {{ $payments->links() }}
            </div>
        @endif
    </div>
@endsection

@push('scripts')
    <script>
        function viewPayment(id) {
            fetch(`/admin/payments/${id}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showPaymentModal(data.payment);
                    }
                });
        }

        function showPaymentModal(payment) {
            const modalHtml = `
        <div class="modal fade" id="viewPaymentModal" tabindex="-1">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Payment Details</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <strong>Transaction ID:</strong> <code>${payment.transaction_id || 'N/A'}</code>
                            </div>
                            <div class="col-md-6">
                                <strong>Business:</strong> ${payment.business?.business_name || 'N/A'}
                            </div>
                        </div>
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <strong>Amount:</strong> ${payment.currency} ${payment.amount}
                            </div>
                            <div class="col-md-6">
                                <strong>Payment Method:</strong> ${payment.payment_method || 'N/A'}
                            </div>
                        </div>
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <strong>Status:</strong> <span class="badge bg-${payment.status === 'completed' ? 'success' : payment.status === 'pending' ? 'warning' : 'danger'}">${payment.status}</span>
                            </div>
                            <div class="col-md-6">
                                <strong>Payment Date:</strong> ${payment.payment_date || 'N/A'}
                            </div>
                        </div>
                        ${payment.subscription ? `
                        <div class="row mb-3">
                            <div class="col-12">
                                <strong>Subscription Plan:</strong> ${payment.subscription.plan?.name || 'N/A'}
                            </div>
                        </div>
                        ` : ''}
                        ${payment.description ? `
                        <div class="row mb-3">
                            <div class="col-12">
                                <strong>Description:</strong> ${payment.description}
                            </div>
                        </div>
                        ` : ''}
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    </div>
                </div>
            </div>
        </div>
    `;

            document.body.insertAdjacentHTML('beforeend', modalHtml);
            const modal = new bootstrap.Modal(document.getElementById('viewPaymentModal'));
            modal.show();

            document.getElementById('viewPaymentModal').addEventListener('hidden.bs.modal', function () {
                this.remove();
            });
        }
    </script>
@endpush
