@extends('layouts.admin')

@section('title', 'Venue Details - ' . $venue->name)

@section('content')
    <div class="admin-header mb-4">
        <div>
            <h2>{{ $venue->name }}</h2>
            <p class="text-muted mb-0">
                Business: {{ $venue->business->business_name }} |
                Category: {{ $venue->category->name }}
            </p>
        </div>
        <div class="d-flex gap-2">
            <button class="btn btn-primary" onclick="approveVenue({{ $venue->id }})">
                <i class="bi bi-check-circle me-2"></i>Approve
            </button>
            <button class="btn btn-danger" onclick="rejectVenue({{ $venue->id }})">
                <i class="bi bi-x-circle me-2"></i>Reject
            </button>
            <a href="{{ route('admin.venues.index') }}" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left me-2"></i>Back
            </a>
        </div>
    </div>

    <!-- Status & Quick Info -->
    <div class="row g-4 mb-4">
        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="text-muted small mb-2">Status</div>
                    @if($venue->status === 'active')
                        <span class="badge bg-success">Active</span>
                    @elseif($venue->status === 'pending')
                        <span class="badge bg-warning">Pending Approval</span>
                    @else
                        <span class="badge bg-secondary">{{ ucfirst($venue->status) }}</span>
                    @endif
                </div>
            </div>
        </div>

        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="text-muted small mb-2">Total Offers</div>
                    <div class="fs-4 fw-bold">{{ $venue->offers()->count() }}</div>
                </div>
            </div>
        </div>

        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="text-muted small mb-2">Reviews</div>
                    <div class="fs-4 fw-bold">
                        {{ $venue->reviews()->count() }}
                        @if($venue->rating)
                            <small class="text-muted fs-6">({{ number_format($venue->rating, 1) }}★)</small>
                        @endif
                    </div>
                </div>
            </div>
        </div>

        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="text-muted small mb-2">Redemptions</div>
                    <div class="fs-4 fw-bold">{{ $venue->redemptions()->count() }}</div>
                </div>
            </div>
        </div>
    </div>

    <div class="row g-4">
        <div class="col-lg-8">
            <!-- Featured Image -->
            @if($venue->featured_image)
                <div class="card border-0 shadow-sm mb-4">
                    <img src="{{ Storage::url($venue->featured_image) }}"
                         class="card-img-top"
                         alt="{{ $venue->name }}"
                         style="height: 400px; object-fit: cover;">
                </div>
            @endif

            <!-- Details -->
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Venue Information</h5>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="text-muted small">Venue Name</label>
                                <div class="fw-semibold">{{ $venue->name }}</div>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="text-muted small">Category</label>
                                <div class="fw-semibold">{{ $venue->category->name }}</div>
                            </div>
                        </div>

                        <div class="col-12">
                            <div class="mb-3">
                                <label class="text-muted small">Description</label>
                                <div>{{ $venue->description ?? 'No description provided' }}</div>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="text-muted small">City</label>
                                <div class="fw-semibold">{{ $venue->city }}</div>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="text-muted small">Area</label>
                                <div class="fw-semibold">{{ $venue->area }}</div>
                            </div>
                        </div>

                        <div class="col-12">
                            <div class="mb-3">
                                <label class="text-muted small">Full Address</label>
                                <div>{{ $venue->address }}</div>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="text-muted small">Phone</label>
                                <div class="fw-semibold">{{ $venue->phone }}</div>
                            </div>
                        </div>

                        @if($venue->email)
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="text-muted small">Email</label>
                                    <div class="fw-semibold">{{ $venue->email }}</div>
                                </div>
                            </div>
                        @endif

                        @if($venue->website)
                            <div class="col-12">
                                <div class="mb-3">
                                    <label class="text-muted small">Website</label>
                                    <div>
                                        <a href="{{ $venue->website }}" target="_blank">{{ $venue->website }}</a>
                                    </div>
                                </div>
                            </div>
                        @endif

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="text-muted small">Coordinates</label>
                                <div>{{ $venue->latitude }}, {{ $venue->longitude }}</div>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="mb-0">
                                <label class="text-muted small">Created</label>
                                <div>{{ $venue->created_at->format('M d, Y h:i A') }}</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Gallery -->
            @if($venue->images && count(json_decode($venue->images, true)) > 0)
                <div class="card border-0 shadow-sm mb-4">
                    <div class="card-header bg-white border-bottom">
                        <h5 class="mb-0 fw-bold">Gallery</h5>
                    </div>
                    <div class="card-body">
                        <div class="row g-3">
                            @foreach(json_decode($venue->images, true) as $image)
                                <div class="col-md-4">
                                    <img src="{{ Storage::url($image) }}"
                                         class="img-fluid rounded"
                                         style="height: 200px; width: 100%; object-fit: cover;">
                                </div>
                            @endforeach
                        </div>
                    </div>
                </div>
            @endif

            <!-- Offers -->
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Active Offers</h5>
                </div>
                <div class="card-body p-0">
                    @php
                        $offers = $venue->offers()->where('status', 'active')->take(10)->get();
                    @endphp

                    @if($offers->count() > 0)
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="bg-light">
                                <tr>
                                    <th>Offer Title</th>
                                    <th>Discount</th>
                                    <th>Valid Until</th>
                                    <th>Redemptions</th>
                                    <th>Status</th>
                                </tr>
                                </thead>
                                <tbody>
                                @foreach($offers as $offer)
                                    <tr>
                                        <td>{{ $offer->title }}</td>
                                        <td>
                                            @if($offer->discount_percentage)
                                                {{ $offer->discount_percentage }}%
                                            @elseif($offer->discount_amount)
                                                AED {{ number_format($offer->discount_amount) }}
                                            @else
                                                -
                                            @endif
                                        </td>
                                        <td>{{ $offer->end_date ? $offer->end_date->format('M d, Y') : '-' }}</td>
                                        <td>{{ $offer->redemption_count }}</td>
                                        <td>
                                            <span class="badge bg-success">Active</span>
                                        </td>
                                    </tr>
                                @endforeach
                                </tbody>
                            </table>
                        </div>
                    @else
                        <div class="text-center py-4 text-muted">
                            No active offers
                        </div>
                    @endif
                </div>
            </div>
        </div>

        <!-- Sidebar -->
        <div class="col-lg-4">
            <!-- Business Info -->
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Business Information</h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label class="text-muted small">Business Name</label>
                        <div class="fw-semibold">{{ $venue->business->business_name }}</div>
                    </div>

                    <div class="mb-3">
                        <label class="text-muted small">Owner</label>
                        <div class="fw-semibold">{{ $venue->business->owner->full_name }}</div>
                    </div>

                    <div class="mb-3">
                        <label class="text-muted small">Email</label>
                        <div>{{ $venue->business->email }}</div>
                    </div>

                    <div class="mb-0">
                        <label class="text-muted small">Phone</label>
                        <div>{{ $venue->business->phone }}</div>
                    </div>
                </div>
            </div>

            <!-- Admin Actions -->
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Admin Actions</h5>
                </div>
                <div class="card-body">
                    @if($venue->status === 'pending')
                        <button class="btn btn-success w-100 mb-2" onclick="approveVenue({{ $venue->id }})">
                            <i class="bi bi-check-circle me-2"></i>Approve Venue
                        </button>
                        <button class="btn btn-danger w-100 mb-2" onclick="rejectVenue({{ $venue->id }})">
                            <i class="bi bi-x-circle me-2"></i>Reject Venue
                        </button>
                    @else
                        <button class="btn btn-warning w-100 mb-2" onclick="toggleStatus({{ $venue->id }})">
                            <i class="bi bi-power me-2"></i>
                            {{ $venue->status === 'active' ? 'Deactivate' : 'Activate' }}
                        </button>
                    @endif

                    <a href="{{ route('admin.businesses.show', $venue->business_id) }}"
                       class="btn btn-outline-primary w-100 mb-2">
                        <i class="bi bi-building me-2"></i>View Business
                    </a>

                    <button class="btn btn-outline-danger w-100" onclick="deleteVenue({{ $venue->id }})">
                        <i class="bi bi-trash me-2"></i>Delete Venue
                    </button>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('scripts')
    <script>
        function approveVenue(id) {
            if (confirm('Approve this venue?')) {
                fetch(`/admin/venues/${id}/approve`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Accept': 'application/json'
                    }
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            location.reload();
                        } else {
                            alert(data.message || 'Error approving venue');
                        }
                    });
            }
        }

        function rejectVenue(id) {
            const reason = prompt('Enter rejection reason:');
            if (reason) {
                fetch(`/admin/venues/${id}/reject`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({ reason: reason })
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            location.reload();
                        } else {
                            alert(data.message || 'Error rejecting venue');
                        }
                    });
            }
        }

        function toggleStatus(id) {
            if (confirm('Change venue status?')) {
                fetch(`/admin/venues/${id}/toggle-status`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Accept': 'application/json'
                    }
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            location.reload();
                        }
                    });
            }
        }

        function deleteVenue(id) {
            if (confirm('Are you sure you want to delete this venue? This action cannot be undone.')) {
                fetch(`/admin/venues/${id}`, {
                    method: 'DELETE',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Accept': 'application/json'
                    }
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            window.location.href = '/admin/venues';
                        } else {
                            alert(data.message || 'Error deleting venue');
                        }
                    });
            }
        }
    </script>
@endpush
