@extends('layouts.business')

@section('title', 'Checkout - ' . $subscription->plan->name)

@push('styles')
    <style>
        #payment-element {
            margin-bottom: 24px;
        }
        .payment-error {
            color: #dc3545;
            font-size: 14px;
            margin-top: 8px;
        }
        .StripeElement {
            box-sizing: border-box;
            height: 40px;
            padding: 10px 12px;
            border: 1px solid #ced4da;
            border-radius: 4px;
            background-color: white;
            transition: border-color 0.15s ease-in-out;
        }
        .StripeElement--focus {
            border-color: #80bdff;
            outline: 0;
            box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
        }
        .StripeElement--invalid {
            border-color: #dc3545;
        }
    </style>
@endpush

@section('content')
    <div class="row justify-content-center">
        <div class="col-lg-10">
            <div class="mb-4">
                <a href="{{ route('business.subscription.plans') }}" class="btn btn-outline-secondary mb-3">
                    <i class="bi bi-arrow-left me-2"></i>Back to Plans
                </a>
                <h2 class="fw-bold mb-1">Complete Your Subscription</h2>
                <p class="text-muted mb-0">Review your order and complete payment</p>
            </div>

            @if(session('success'))
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    {{ session('success') }}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            @endif

            @if(session('error'))
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    {{ session('error') }}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            @endif

            <div class="row g-4">
                <!-- Payment Form -->
                <div class="col-lg-8">
                    <!-- Plan Details -->
                    <div class="card border-0 shadow-sm mb-4">
                        <div class="card-header bg-white border-bottom">
                            <h5 class="mb-0 fw-bold">Plan Details</h5>
                        </div>
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-start mb-3">
                                <div>
                                    <h4 class="mb-1">{{ $subscription->plan->name }}</h4>
                                    @if($subscription->plan->name_ar)
                                        <p class="text-muted mb-2">{{ $subscription->plan->name_ar }}</p>
                                    @endif
                                    <p class="mb-0">{{ $subscription->plan->description }}</p>
                                </div>
                                <div class="text-end">
                                    <h3 class="mb-0 text-primary">{{ $payment->currency }} {{ number_format($payment->amount, 2) }}</h3>
                                    <small class="text-muted">per {{ $subscription->plan->duration }}</small>
                                </div>
                            </div>

                            <hr>

                            <h6 class="fw-bold mb-3">Plan Features:</h6>
                            <div class="row g-3">
                                @if($subscription->plan->max_venues)
                                    <div class="col-md-6">
                                        <div class="d-flex align-items-center">
                                            <i class="bi bi-geo-alt text-primary me-2"></i>
                                            <span>{{ $subscription->plan->max_venues }} Venues</span>
                                        </div>
                                    </div>
                                @endif
                                @if($subscription->plan->max_offers)
                                    <div class="col-md-6">
                                        <div class="d-flex align-items-center">
                                            <i class="bi bi-tag text-primary me-2"></i>
                                            <span>{{ $subscription->plan->max_offers }} Offers</span>
                                        </div>
                                    </div>
                                @endif
                                @if($subscription->plan->max_staff)
                                    <div class="col-md-6">
                                        <div class="d-flex align-items-center">
                                            <i class="bi bi-people text-primary me-2"></i>
                                            <span>{{ $subscription->plan->max_staff }} Staff Members</span>
                                        </div>
                                    </div>
                                @endif
                            </div>

                            @if($subscription->plan->features)
                                @php
                                    $features = is_array($subscription->plan->features) ? $subscription->plan->features : json_decode($subscription->plan->features, true);
                                @endphp
                                @if($features)
                                    <hr>
                                    <h6 class="fw-bold mb-3">Additional Features:</h6>
                                    <div class="row g-2">
                                        @foreach($features as $feature)
                                            <div class="col-md-6">
                                                <div class="d-flex align-items-center">
                                                    <i class="bi bi-check-circle-fill text-success me-2"></i>
                                                    <span>{{ is_array($feature) ? $feature['name'] ?? $feature[0] : $feature }}</span>
                                                </div>
                                            </div>
                                        @endforeach
                                    </div>
                                @endif
                            @endif
                        </div>
                    </div>

                    <!-- Payment Method -->
                    <div class="card border-0 shadow-sm mb-4">
                        <div class="card-header bg-white border-bottom">
                            <h5 class="mb-0 fw-bold">Payment Method</h5>
                        </div>
                        <div class="card-body">
                            <!-- Payment Method Selection -->
                            <div class="mb-4">
                                <label class="form-label fw-semibold">Select Payment Gateway</label>
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <div class="form-check card-payment-option">
                                            <input class="form-check-input" type="radio" name="payment_gateway" id="stripe" value="stripe" checked>
                                            <label class="form-check-label w-100" for="stripe">
                                                <div class="p-3 border rounded" style="cursor: pointer;">
                                                    <i class="bi bi-credit-card fs-4 text-primary"></i>
                                                    <div class="mt-2">
                                                        <strong>Credit/Debit Card</strong>
                                                        <div class="small text-muted">Secure payment via Stripe</div>
                                                    </div>
                                                </div>
                                            </label>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-check card-payment-option">
                                            <input class="form-check-input" type="radio" name="payment_gateway" id="paypal" value="paypal">
                                            <label class="form-check-label w-100" for="paypal">
                                                <div class="p-3 border rounded" style="cursor: pointer;">
                                                    <i class="bi bi-paypal fs-4" style="color: #00457C;"></i>
                                                    <div class="mt-2">
                                                        <strong>PayPal</strong>
                                                        <div class="small text-muted">Pay with PayPal account</div>
                                                    </div>
                                                </div>
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Stripe Payment Form -->
                            <div id="stripe-payment-section">
                                <form id="payment-form">
                                    @csrf
                                    <div id="payment-element" class="mb-3">
                                        <!-- Stripe Payment Element will be inserted here -->
                                    </div>

                                    <div id="payment-message" class="payment-error" style="display:none;"></div>

                                    <!-- Terms -->
                                    <div class="form-check mb-4">
                                        <input class="form-check-input" type="checkbox" id="terms" required>
                                        <label class="form-check-label" for="terms">
                                            I agree to the <a href="#" target="_blank">Terms of Service</a> and <a href="#" target="_blank">Privacy Policy</a>
                                        </label>
                                    </div>

                                    <div class="alert alert-info">
                                        <i class="bi bi-info-circle me-2"></i>
                                        <strong>Note:</strong> Your subscription will be activated immediately after successful payment. You can cancel anytime.
                                    </div>

                                    <button type="submit" class="btn btn-primary btn-lg w-100" id="submit-button">
                                    <span id="button-text">
                                        <i class="bi bi-lock-fill me-2"></i>
                                        Pay {{ $payment->currency }} {{ number_format($payment->amount, 2) }}
                                    </span>
                                        <span id="button-spinner" class="spinner-border spinner-border-sm me-2" style="display:none;"></span>
                                    </button>
                                </form>
                            </div>

                            <!-- PayPal Payment Section -->
                            <div id="paypal-payment-section" style="display: none;">
                                <div class="alert alert-info">
                                    <i class="bi bi-info-circle me-2"></i>
                                    You will be redirected to PayPal to complete your payment.
                                </div>
                                <div id="paypal-button-container"></div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Order Summary Sidebar -->
                <div class="col-lg-4">
                    <div class="card border-0 shadow-sm sticky-top" style="top: 20px;">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0">Order Summary</h5>
                        </div>
                        <div class="card-body">
                            <div class="d-flex justify-content-between mb-2">
                                <span>Plan:</span>
                                <strong>{{ $subscription->plan->name }}</strong>
                            </div>
                            <div class="d-flex justify-content-between mb-2">
                                <span>Billing:</span>
                                <strong>{{ ucfirst($subscription->plan->duration) }}</strong>
                            </div>

                            <hr>

                            <div class="d-flex justify-content-between mb-2">
                                <span>Subtotal:</span>
                                <strong>{{ $payment->currency }} {{ number_format($payment->amount, 2) }}</strong>
                            </div>

                            <hr>

                            <div class="d-flex justify-content-between">
                                <span class="fw-bold">Total:</span>
                                <strong class="text-primary fs-4">{{ $payment->currency }} {{ number_format($payment->amount, 2) }}</strong>
                            </div>

                            <div class="mt-4 p-3 bg-light rounded">
                                <small class="text-muted">
                                    <i class="bi bi-shield-check me-2"></i>
                                    Your payment information is secure and encrypted
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('scripts')
    <script src="https://js.stripe.com/v3/"></script>
    <script>
        // Check if Stripe key is configured
        const stripeKey = '{{ config('services.stripe.key') }}';

        if (!stripeKey || stripeKey === '') {
            document.addEventListener('DOMContentLoaded', function() {
                document.getElementById('payment-element').innerHTML = `
                <div class="alert alert-danger">
                    <h5><i class="bi bi-exclamation-triangle me-2"></i>Stripe Not Configured</h5>
                    <p class="mb-2"><strong>Payment service is not available.</strong></p>
                    <p class="mb-2">The administrator needs to configure Stripe API keys.</p>
                    <hr>
                    <p class="mb-1"><strong>Quick Setup:</strong></p>
                    <ol class="mb-2">
                        <li>Get Stripe API keys from <a href="https://dashboard.stripe.com/register" target="_blank">Stripe Dashboard</a></li>
                        <li>Add to <code>.env</code> file:
                            <pre class="bg-dark text-light p-2 mt-2 rounded">STRIPE_KEY=pk_test_your_key_here
STRIPE_SECRET=sk_test_your_key_here</pre>
                        </li>
                        <li>Run: <code>php artisan config:clear && php artisan cache:clear</code></li>
                        <li>Restart server and refresh this page</li>
                    </ol>
                    <p class="mb-0">
                        <strong>📖 See:</strong> <code>STRIPE_SETUP_GUIDE.md</code> for complete instructions
                    </p>
                </div>
            `;
                document.getElementById('submit-button').disabled = true;
            });
            throw new Error('Stripe not configured');
        }

        const stripe = Stripe(stripeKey);
        const paymentId = '{{ $payment->id }}';
        let elements;
        let paymentElement;

        // Initialize Stripe
        async function initializeStripe() {
            try {
                // Create payment intent
                const response = await fetch('{{ route('stripe.create-intent') }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({
                        payment_id: paymentId
                    })
                });

                const data = await response.json();

                if (!data.success) {
                    showMessage(data.message || 'Failed to initialize payment');
                    return;
                }

                // Create Elements instance
                elements = stripe.elements({
                    clientSecret: data.client_secret
                });

                // Create and mount Payment Element
                paymentElement = elements.create('payment');
                paymentElement.mount('#payment-element');

                paymentElement.on('ready', function() {
                    console.log('Payment Element ready');
                });

                paymentElement.on('change', function(event) {
                    if (event.error) {
                        showMessage(event.error.message);
                    } else {
                        showMessage('');
                    }
                });

            } catch (error) {
                console.error('Stripe initialization error:', error);
                showMessage('Failed to load payment form. Please refresh the page.');
            }
        }

        // Handle form submission
        document.getElementById('payment-form').addEventListener('submit', async function(e) {
            e.preventDefault();

            if (!document.getElementById('terms').checked) {
                showMessage('Please accept the terms and conditions');
                return;
            }

            setLoading(true);

            try {
                const {error, paymentIntent} = await stripe.confirmPayment({
                    elements,
                    confirmParams: {
                        return_url: '{{ route('business.subscription.index') }}',
                    },
                    redirect: 'if_required'
                });

                if (error) {
                    showMessage(error.message);
                    setLoading(false);
                } else if (paymentIntent && paymentIntent.status === 'succeeded') {
                    // Process successful payment
                    await processPayment(paymentIntent.id);
                }
            } catch (error) {
                console.error('Payment error:', error);
                showMessage('Payment failed. Please try again.');
                setLoading(false);
            }
        });

        // Process payment on server
        async function processPayment(paymentIntentId) {
            try {
                const response = await fetch('{{ route('stripe.process') }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({
                        payment_intent_id: paymentIntentId,
                        payment_id: paymentId
                    })
                });

                const data = await response.json();

                if (data.success) {
                    // Redirect to success page
                    window.location.href = '{{ route('business.subscription.index') }}?payment=success';
                } else {
                    showMessage(data.message || 'Payment processing failed');
                    setLoading(false);
                }
            } catch (error) {
                console.error('Process payment error:', error);
                showMessage('Failed to process payment. Please contact support.');
                setLoading(false);
            }
        }

        // Show message
        function showMessage(messageText) {
            const messageContainer = document.querySelector('#payment-message');
            messageContainer.textContent = messageText;
            messageContainer.style.display = messageText ? 'block' : 'none';
        }

        // Toggle loading state
        function setLoading(isLoading) {
            const submitButton = document.querySelector('#submit-button');
            const buttonText = document.querySelector('#button-text');
            const spinner = document.querySelector('#button-spinner');

            if (isLoading) {
                submitButton.disabled = true;
                buttonText.style.display = 'none';
                spinner.style.display = 'inline-block';
            } else {
                submitButton.disabled = false;
                buttonText.style.display = 'inline';
                spinner.style.display = 'none';
            }
        }

        // Payment gateway switcher
        document.querySelectorAll('input[name="payment_gateway"]').forEach(radio => {
            radio.addEventListener('change', function() {
                if (this.value === 'stripe') {
                    document.getElementById('stripe-payment-section').style.display = 'block';
                    document.getElementById('paypal-payment-section').style.display = 'none';
                } else if (this.value === 'paypal') {
                    document.getElementById('stripe-payment-section').style.display = 'none';
                    document.getElementById('paypal-payment-section').style.display = 'block';
                    // Initialize PayPal (if needed)
                    // initializePayPal();
                }
            });
        });

        // Initialize Stripe on page load
        document.addEventListener('DOMContentLoaded', function() {
            initializeStripe();
        });
    </script>
@endpush
