<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\API\V1\Auth\{AuthController, LoginController, RegisterController, SocialAuthController, OtpController};
use App\Http\Controllers\API\V1\Business\{BusinessRegistrationController, OfferManagementController, VenueManagementController};
use App\Http\Controllers\API\V1\User\ProfileController;
use App\Http\Controllers\API\V1\Venue\VenueController;
use App\Http\Controllers\API\V1\Offer\{OfferController, RedemptionController};
use App\Http\Controllers\API\V1\Spotlight\SpotlightController;
use App\Http\Controllers\API\V1\Subscription\SubscriptionController;
use App\Http\Controllers\API\V1\GDPR\GdprController;
use App\Http\Controllers\API\V1\Marketing\MarketingController;

/*
|--------------------------------------------------------------------------
| API Routes - Privasee Platform - COMPLETE
|--------------------------------------------------------------------------
*/

// Public Configuration
Route::get('config/public', [\App\Http\Controllers\Admin\ConfigurationController::class, 'getPublic']);

// Public routes
Route::prefix('v1')->group(function () {

    // ========== AUTHENTICATION ==========
    Route::prefix('auth')->group(function () {
        // Email/Password Auth
        Route::post('register', [RegisterController::class, 'register']);
        Route::post('login', [LoginController::class, 'login']);
        Route::post('forgot-password', [AuthController::class, 'forgotPassword']);
        Route::post('reset-password', [AuthController::class, 'resetPassword']);

        // Social Login (OAuth)
        Route::get('google', [SocialAuthController::class, 'redirectToGoogle']);
        Route::get('google/callback', [SocialAuthController::class, 'handleGoogleCallback']);
        Route::get('facebook', [SocialAuthController::class, 'redirectToFacebook']);
        Route::get('facebook/callback', [SocialAuthController::class, 'handleFacebookCallback']);

        // OTP System
        Route::post('otp/send', [OtpController::class, 'sendOtp']);
        Route::post('otp/verify', [OtpController::class, 'verifyOtp']);
        Route::post('otp/resend', [OtpController::class, 'resendOtp']);
        Route::post('otp/verify-login', [OtpController::class, 'verifyAndLogin']);
    });

    // ========== BUSINESS REGISTRATION (PUBLIC) ==========
    Route::prefix('business')->group(function () {
        Route::post('register', [BusinessRegistrationController::class, 'register']);
        Route::post('check-availability', [BusinessRegistrationController::class, 'checkAvailability']);
    });

    // ========== PUBLIC BROWSING ==========
    // Categories
    Route::get('categories', [\App\Http\Controllers\API\V1\Category\CategoryController::class, 'index']);
    Route::get('categories/{id}', [\App\Http\Controllers\API\V1\Category\CategoryController::class, 'show']);

    // Venues (Public)
    Route::get('venues', [VenueController::class, 'index']);
    Route::get('venues/{id}', [VenueController::class, 'show']);
    Route::get('venues/category/{categoryId}', [VenueController::class, 'byCategory']);
    Route::get('venues/search', [VenueController::class, 'search']);
    Route::get('venues/{id}/offers', [VenueController::class, 'offers']);
    Route::get('venues/{id}/reviews', [VenueController::class, 'reviews']);

    // Offers (Public)
    Route::get('offers', [OfferController::class, 'index']);
    Route::get('offers/{id}', [OfferController::class, 'show']);
    Route::get('offers/search', [OfferController::class, 'search']);
    Route::get('offers/exclusive', [OfferController::class, 'exclusive']);
    Route::get('offers/special', [OfferController::class, 'special']);
    Route::get('offers/featured', [OfferController::class, 'featured']);

    // Subscription Plans (Public)
    Route::get('subscription-plans', [SubscriptionController::class, 'plans']);

    // Spotlights (Public Feed)
    Route::get('spotlights', [SpotlightController::class, 'index']);
    Route::get('spotlights/{id}', [SpotlightController::class, 'show']);

    // Marketing Content (Public)
    Route::prefix('marketing')->group(function () {
        Route::get('banners', [MarketingController::class, 'activeBanners']);
        Route::get('popups', [MarketingController::class, 'activePopups']);
        Route::get('featured-venues', [MarketingController::class, 'featuredVenues']);
        Route::get('featured-offers', [MarketingController::class, 'featuredOffers']);
        Route::post('click/{id}', [MarketingController::class, 'trackClick']);
        Route::post('conversion/{id}', [MarketingController::class, 'trackConversion']);
    });

    // FAQ (Public)
    Route::get('faqs', [\App\Http\Controllers\API\V1\FAQ\FaqController::class, 'index']);
    Route::get('faqs/category/{category}', [\App\Http\Controllers\API\V1\FAQ\FaqController::class, 'byCategory']);
});

// ========== PROTECTED ROUTES (Authenticated Users) ==========
Route::prefix('v1')->middleware('auth:sanctum')->group(function () {

    // ========== AUTH & PROFILE ==========
    Route::prefix('auth')->group(function () {
        Route::post('logout', [AuthController::class, 'logout']);
        Route::post('logout-all', [AuthController::class, 'logoutAll']);
        Route::get('me', [AuthController::class, 'me']);
        Route::put('password', [AuthController::class, 'changePassword']);
    });

    // User Profile
    Route::prefix('profile')->group(function () {
        Route::get('/', [ProfileController::class, 'show']);
        Route::put('/', [ProfileController::class, 'update']);
        Route::post('image', [ProfileController::class, 'uploadImage']);
        Route::delete('image', [ProfileController::class, 'deleteImage']);
        Route::put('preferences', [ProfileController::class, 'updatePreferences']);
        Route::get('statistics', [ProfileController::class, 'statistics']);
    });

    // Social Accounts Management
    Route::prefix('social-accounts')->group(function () {
        Route::get('/', [SocialAuthController::class, 'linkedAccounts']);
        Route::post('link', [SocialAuthController::class, 'linkSocialAccount']);
        Route::delete('{provider}', [SocialAuthController::class, 'unlinkSocialAccount']);
    });

    // ========== SUBSCRIPTIONS ==========
    Route::prefix('subscriptions')->group(function () {
        Route::get('plans', [SubscriptionController::class, 'plans']);
        Route::get('current', [SubscriptionController::class, 'current']);
        Route::post('subscribe', [SubscriptionController::class, 'subscribe']);
        Route::post('cancel', [SubscriptionController::class, 'cancel']);
        Route::post('renew', [SubscriptionController::class, 'renew']);
        Route::get('history', [SubscriptionController::class, 'history']);
    });

    // ========== VENUES ==========
    Route::prefix('venues')->group(function () {
        Route::post('{id}/favorite', [VenueController::class, 'toggleFavorite']);
        Route::post('{id}/review', [VenueController::class, 'addReview']);
        Route::put('reviews/{reviewId}', [VenueController::class, 'updateReview']);
        Route::delete('reviews/{reviewId}', [VenueController::class, 'deleteReview']);
    });

    // Favorites
    Route::get('favorites', [ProfileController::class, 'favorites']);

    // ========== OFFERS ==========
    Route::prefix('offers')->group(function () {
        Route::post('{id}/favorite', [OfferController::class, 'toggleFavorite']);
    });

    // ========== REDEMPTIONS ==========
    Route::prefix('redemptions')->group(function () {
        Route::post('redeem/{offerId}', [RedemptionController::class, 'redeem']);
        Route::get('/', [RedemptionController::class, 'myRedemptions']);
        Route::get('{id}', [RedemptionController::class, 'show']);
        Route::post('{id}/rate', [RedemptionController::class, 'rate']);
    });

    // ========== SPOTLIGHTS (SOCIAL FEED) ==========
    Route::prefix('spotlights')->group(function () {
        Route::post('/', [SpotlightController::class, 'store']);
        Route::put('{id}', [SpotlightController::class, 'update']);
        Route::delete('{id}', [SpotlightController::class, 'destroy']);
        Route::post('{id}/like', [SpotlightController::class, 'toggleLike']);
        Route::post('{id}/comment', [SpotlightController::class, 'comment']);
        Route::put('comments/{commentId}', [SpotlightController::class, 'updateComment']);
        Route::delete('comments/{commentId}', [SpotlightController::class, 'deleteComment']);
        Route::post('{id}/flag', [SpotlightController::class, 'flag']);
    });

    // ========== REFERRALS ==========
    Route::prefix('referrals')->group(function () {
        Route::get('my-code', [\App\Http\Controllers\API\V1\Referral\ReferralController::class, 'myCode']);
        Route::get('my-referrals', [\App\Http\Controllers\API\V1\Referral\ReferralController::class, 'myReferrals']);
        Route::get('statistics', [\App\Http\Controllers\API\V1\Referral\ReferralController::class, 'statistics']);
    });

    // ========== PAYMENTS ==========
    Route::prefix('payments')->group(function () {
        Route::get('history', [\App\Http\Controllers\API\V1\Payment\PaymentController::class, 'history']);
        Route::get('{id}', [\App\Http\Controllers\API\V1\Payment\PaymentController::class, 'show']);
    });

    // ========== SUPPORT ==========
    Route::prefix('support')->group(function () {
        Route::get('tickets', [\App\Http\Controllers\API\V1\Support\SupportController::class, 'index']);
        Route::post('tickets', [\App\Http\Controllers\API\V1\Support\SupportController::class, 'create']);
        Route::get('tickets/{id}', [\App\Http\Controllers\API\V1\Support\SupportController::class, 'show']);
        Route::post('tickets/{id}/reply', [\App\Http\Controllers\API\V1\Support\SupportController::class, 'reply']);
        Route::post('tickets/{id}/close', [\App\Http\Controllers\API\V1\Support\SupportController::class, 'close']);
    });

    // ========== NOTIFICATIONS ==========
    Route::prefix('notifications')->group(function () {
        Route::get('/', [\App\Http\Controllers\API\V1\Notification\NotificationController::class, 'index']);
        Route::post('{id}/read', [\App\Http\Controllers\API\V1\Notification\NotificationController::class, 'markAsRead']);
        Route::post('read-all', [\App\Http\Controllers\API\V1\Notification\NotificationController::class, 'markAllAsRead']);
        Route::delete('{id}', [\App\Http\Controllers\API\V1\Notification\NotificationController::class, 'delete']);
        Route::get('unread-count', [\App\Http\Controllers\API\V1\Notification\NotificationController::class, 'unreadCount']);
    });

    // ========== GDPR ==========
    Route::prefix('gdpr')->group(function () {
        Route::post('request/export', [GdprController::class, 'requestDataExport']);
        Route::post('request/deletion', [GdprController::class, 'requestDataDeletion']);
        Route::post('request/rectification', [GdprController::class, 'requestDataRectification']);
        Route::post('request/access', [GdprController::class, 'requestDataAccess']);
        Route::get('requests', [GdprController::class, 'myRequests']);
        Route::get('download/{id}', [GdprController::class, 'downloadExport']);
        Route::delete('requests/{id}', [GdprController::class, 'cancelRequest']);
    });

    // ========== MARKETING (Business API) ==========
    Route::prefix('marketing')->group(function () {
        Route::get('packages', [MarketingController::class, 'getPackages']);
        Route::post('campaigns', [MarketingController::class, 'createCampaign']);
        Route::get('campaigns', [MarketingController::class, 'myCampaigns']);
        Route::get('campaigns/{id}/analytics', [MarketingController::class, 'getCampaignAnalytics']);
    });
});

// ========== BUSINESS PORTAL API ==========
Route::prefix('v1/business')->middleware(['auth:sanctum'])->group(function () {

    // Dashboard & Status
    Route::get('dashboard', [\App\Http\Controllers\Business\DashboardController::class, 'dashboard']);
    Route::get('status', [BusinessRegistrationController::class, 'status']);
    Route::put('profile', [BusinessRegistrationController::class, 'updateProfile']);

    // Venue Management
    Route::prefix('venues')->group(function () {
        Route::get('/', [VenueManagementController::class, 'index']);
        Route::post('/', [VenueManagementController::class, 'store']);
        Route::get('{id}', [VenueManagementController::class, 'show']);
        Route::put('{id}', [VenueManagementController::class, 'update']);
        Route::delete('{id}', [VenueManagementController::class, 'destroy']);
        Route::get('{id}/statistics', [VenueManagementController::class, 'statistics']);
    });

    // Offer Management
    Route::prefix('offers')->group(function () {
        Route::get('/', [OfferManagementController::class, 'index']);
        Route::post('/', [OfferManagementController::class, 'store']);
        Route::get('{id}', [OfferManagementController::class, 'show']);
        Route::put('{id}', [OfferManagementController::class, 'update']);
        Route::delete('{id}', [OfferManagementController::class, 'destroy']);
        Route::get('{id}/statistics', [OfferManagementController::class, 'statistics']);
        Route::post('{id}/duplicate', [OfferManagementController::class, 'duplicate']);
    });

    // Team Management
    Route::prefix('team')->group(function () {
        Route::get('/', [\App\Http\Controllers\Business\TeamController::class, 'index']);
        Route::post('/', [\App\Http\Controllers\Business\TeamController::class, 'store']);
        Route::get('{id}', [\App\Http\Controllers\Business\TeamController::class, 'show']);
        Route::put('{id}', [\App\Http\Controllers\Business\TeamController::class, 'update']);
        Route::delete('{id}', [\App\Http\Controllers\Business\TeamController::class, 'destroy']);
        Route::post('{id}/resend-invitation', [\App\Http\Controllers\Business\TeamController::class, 'resendInvitation']);
    });

    // Analytics
    Route::get('analytics', [\App\Http\Controllers\Business\AnalyticsController::class, 'index']);
    Route::get('analytics/redemptions', [\App\Http\Controllers\Business\AnalyticsController::class, 'redemptions']);
    Route::get('analytics/revenue', [\App\Http\Controllers\Business\AnalyticsController::class, 'revenue']);

    // Redemptions
    Route::prefix('redemptions')->group(function () {
        Route::get('/', [\App\Http\Controllers\Business\RedemptionController::class, 'index']);
        Route::post('validate/{code}', [\App\Http\Controllers\Business\RedemptionController::class, 'validate']);
        Route::post('scan', [\App\Http\Controllers\Business\RedemptionController::class, 'scanQR']);
    });
});

// ========== ADMIN PANEL API ==========
Route::prefix('v1/admin')->middleware(['auth:sanctum', 'role:admin,super_admin'])->group(function () {

    // Dashboard
    Route::get('dashboard', [\App\Http\Controllers\Admin\DashboardController::class, 'index']);
    Route::get('statistics', [\App\Http\Controllers\Admin\DashboardController::class, 'statistics']);

    // User Management
    Route::prefix('users')->group(function () {
        Route::get('/', [\App\Http\Controllers\Admin\UserController::class, 'index']);
        Route::get('{id}', [\App\Http\Controllers\Admin\UserController::class, 'show']);
        Route::put('{id}', [\App\Http\Controllers\Admin\UserController::class, 'update']);
        Route::delete('{id}', [\App\Http\Controllers\Admin\UserController::class, 'destroy']);
        Route::post('{id}/ban', [\App\Http\Controllers\Admin\UserController::class, 'ban']);
        Route::post('{id}/unban', [\App\Http\Controllers\Admin\UserController::class, 'unban']);
    });

    // Business Management
    Route::prefix('businesses')->group(function () {
        Route::get('/', [\App\Http\Controllers\Admin\BusinessController::class, 'index']);
        Route::get('{id}', [\App\Http\Controllers\Admin\BusinessController::class, 'show']);
        Route::post('{id}/verify', [\App\Http\Controllers\Admin\BusinessController::class, 'verify']);
        Route::post('{id}/reject', [\App\Http\Controllers\Admin\BusinessController::class, 'reject']);
        Route::delete('{id}', [\App\Http\Controllers\Admin\BusinessController::class, 'destroy']);
    });

    // Venue Approval
    Route::prefix('venues')->group(function () {
        Route::get('pending', [\App\Http\Controllers\Admin\VenueController::class, 'pending']);
        Route::post('{id}/approve', [\App\Http\Controllers\Admin\VenueController::class, 'approve']);
        Route::post('{id}/reject', [\App\Http\Controllers\Admin\VenueController::class, 'reject']);
    });

    // Offer Approval
    Route::prefix('offers')->group(function () {
        Route::get('pending', [\App\Http\Controllers\Admin\OfferController::class, 'pending']);
        Route::post('{id}/approve', [\App\Http\Controllers\Admin\OfferController::class, 'approve']);
        Route::post('{id}/reject', [\App\Http\Controllers\Admin\OfferController::class, 'reject']);
    });

    // Spotlight Moderation
    Route::prefix('spotlights')->group(function () {
        Route::get('/', [\App\Http\Controllers\Admin\SpotlightController::class, 'index']);
        Route::get('flagged', [\App\Http\Controllers\Admin\SpotlightController::class, 'flagged']);
        Route::post('{id}/approve', [\App\Http\Controllers\Admin\SpotlightController::class, 'approve']);
        Route::post('{id}/remove', [\App\Http\Controllers\Admin\SpotlightController::class, 'remove']);
    });

    // GDPR Management
    Route::prefix('gdpr')->group(function () {
        Route::get('requests', [\App\Http\Controllers\Admin\GdprController::class, 'index']);
        Route::get('requests/{id}', [\App\Http\Controllers\Admin\GdprController::class, 'show']);
        Route::post('requests/{id}/process', [\App\Http\Controllers\Admin\GdprController::class, 'process']);
        Route::post('requests/{id}/export', [\App\Http\Controllers\Admin\GdprController::class, 'generateExport']);
        Route::post('requests/{id}/access', [\App\Http\Controllers\Admin\GdprController::class, 'completeAccessRequest']);
        Route::post('requests/{id}/rectification', [\App\Http\Controllers\Admin\GdprController::class, 'completeRectification']);
        Route::post('requests/{id}/delete', [\App\Http\Controllers\Admin\GdprController::class, 'deleteUserData']);
        Route::post('requests/{id}/reject', [\App\Http\Controllers\Admin\GdprController::class, 'reject']);
    });

    // System Configuration
    Route::prefix('configuration')->middleware('role:super_admin')->group(function () {
        Route::get('/', [\App\Http\Controllers\Admin\ConfigurationController::class, 'index']);
        Route::get('{key}', [\App\Http\Controllers\Admin\ConfigurationController::class, 'show']);
        Route::put('{key}', [\App\Http\Controllers\Admin\ConfigurationController::class, 'update']);
        Route::get('group/{group}', [\App\Http\Controllers\Admin\ConfigurationController::class, 'byGroup']);
    });

    // Marketing Management
    Route::prefix('marketing')->group(function () {
        Route::get('/', [\App\Http\Controllers\Admin\MarketingController::class, 'index']);
        Route::get('packages', [\App\Http\Controllers\Admin\MarketingController::class, 'packages']);
        Route::post('packages', [\App\Http\Controllers\Admin\MarketingController::class, 'createPackage']);
        Route::put('packages/{id}', [\App\Http\Controllers\Admin\MarketingController::class, 'updatePackage']);
        Route::delete('packages/{id}', [\App\Http\Controllers\Admin\MarketingController::class, 'deletePackage']);
        Route::get('campaigns', [\App\Http\Controllers\Admin\MarketingController::class, 'campaigns']);
        Route::get('campaigns/{id}', [\App\Http\Controllers\Admin\MarketingController::class, 'viewCampaign']);
        Route::post('campaigns/{id}/approve', [\App\Http\Controllers\Admin\MarketingController::class, 'approveCampaign']);
        Route::post('campaigns/{id}/reject', [\App\Http\Controllers\Admin\MarketingController::class, 'rejectCampaign']);
        Route::get('analytics', [\App\Http\Controllers\Admin\MarketingController::class, 'analytics']);
    });

    // Analytics
    Route::get('analytics', [\App\Http\Controllers\Admin\DashboardController::class, 'analytics']);
    Route::get('analytics/revenue', [\App\Http\Controllers\Admin\DashboardController::class, 'revenue']);
    Route::get('analytics/users', [\App\Http\Controllers\Admin\DashboardController::class, 'users']);

    // Audit Logs
    Route::get('audit-logs', [\App\Http\Controllers\Admin\AuditController::class, 'index']);
    Route::get('audit-logs/{id}', [\App\Http\Controllers\Admin\AuditController::class, 'show']);

    // FAQ Management
    Route::prefix('faqs')->group(function () {
        Route::get('/', [\App\Http\Controllers\Admin\FaqController::class, 'index']);
        Route::post('/', [\App\Http\Controllers\Admin\FaqController::class, 'store']);
        Route::get('{id}', [\App\Http\Controllers\Admin\FaqController::class, 'show']);
        Route::put('{id}', [\App\Http\Controllers\Admin\FaqController::class, 'update']);
        Route::delete('{id}', [\App\Http\Controllers\Admin\FaqController::class, 'destroy']);
        Route::post('{id}/toggle', [\App\Http\Controllers\Admin\FaqController::class, 'toggle']);
    });
});
