<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Payment\WebhookController;
use App\Http\Controllers\Business\{
    DashboardController as BusinessDashboardController,
    TeamController,
    AuthController as BusinessAuthController,
    RegistrationController as BusinessRegistrationController,
    VenueController as BusinessVenueController,
    OfferController as BusinessOfferController,
    RedemptionController as BusinessRedemptionController,
    SubscriptionController as BusinessSubscriptionController,
    SettingsController as BusinessSettingsController,
    AnalyticsController as BusinessAnalyticsController,
    MarketingController as BusinessMarketingController,
    NotificationController as BusinessNotificationController,
    ReferralController as BusinessReferralController,
    SupportController as BusinessSupportController
};
use App\Http\Controllers\Admin\{
    DashboardController as AdminDashboardController,
    AuthController as AdminAuthController,
    UserController as AdminUserController,
    BusinessController as AdminBusinessController,
    CategoryController as AdminCategoryController,
    SettingsController as AdminSettingsController,
    SpotlightController as AdminSpotlightController,
    SubscriptionPlanController as AdminSubscriptionPlanController,
    VenueController as AdminVenueController,
    OfferController as AdminOfferController,
    AuditController as AdminAuditController,
    ConfigurationController as AdminConfigurationController,
    FaqController as AdminFaqController,
    GdprController as AdminGdprController,
    MarketingController as AdminMarketingController
};

/*
|--------------------------------------------------------------------------
| Web Routes - Privasee Platform
|--------------------------------------------------------------------------
*/

// Landing Page
Route::get('/', function () {
    $userPlans = \App\Models\SubscriptionPlan::where('type', 'user')
        ->where('is_active', true)
        ->orderBy('price', 'asc')
        ->get();

    $businessPlans = \App\Models\SubscriptionPlan::where('type', 'business')
        ->where('is_active', true)
        ->orderBy('price', 'asc')
        ->get();

    return view('welcome', compact('userPlans', 'businessPlans'));
})->name('home');

// Payment Webhooks (must be before any middleware that blocks them)
Route::post('/webhooks/stripe', [WebhookController::class, 'stripe'])->name('webhooks.stripe');
Route::post('/webhooks/paypal', [WebhookController::class, 'paypal'])->name('webhooks.paypal');

// Payment Routes (Public - AJAX endpoints)
Route::prefix('payment')->name('stripe.')->group(function () {
    Route::post('/stripe/create-intent', [\App\Http\Controllers\Payment\StripeController::class, 'createIntent'])->name('create-intent');
    Route::post('/stripe/process', [\App\Http\Controllers\Payment\StripeController::class, 'process'])->name('process');
});

// User Authentication Routes (API-based, but we need named routes for redirects)
Route::get('/login', function () {
    return redirect('/business/login');
})->name('login');

Route::get('/register', function () {
    return redirect('/business/register');
})->name('register');
// Team Invitation Acceptance Routes

Route::get('business/team/accept/{id}', [TeamController::class, 'accept'])->name('business.team.accept-invitation');
Route::post('business/team/accept/{id}', [TeamController::class, 'complete'])->name('business.team.accept.complete');
// Business Authentication Routes (Public)
Route::prefix('business')->name('business.')->group(function () {
    // Registration
    Route::get('/register', [BusinessRegistrationController::class, 'showRegistrationForm'])->name('register');
    Route::post('/register', [BusinessRegistrationController::class, 'register'])->name('register.submit');
    Route::post('/check-availability', [BusinessRegistrationController::class, 'checkAvailability'])->name('check-availability');

    // Login
    Route::get('/login', [BusinessAuthController::class, 'showLoginForm'])->name('login');
    Route::post('/login', [BusinessAuthController::class, 'login'])->name('login.submit');

    // Password Reset
    Route::get('/forgot-password', [BusinessAuthController::class, 'showPasswordResetForm'])->name('password.request');
    Route::post('/forgot-password', [BusinessAuthController::class, 'sendPasswordResetLink'])->name('password.email');


});
// Logout
Route::post('/logout', [BusinessAuthController::class, 'logout'])->name('logout');
// Business Portal Routes (Protected)
Route::prefix('business')->name('business.')->middleware(['auth'])->group(function () {
    // Dashboard
    Route::get('/dashboard', [BusinessDashboardController::class, 'index'])->name('dashboard');

    // Analytics
    Route::get('/analytics', [BusinessAnalyticsController::class, 'index'])->name('analytics');
    Route::get('/analytics/data', [BusinessAnalyticsController::class, 'getData'])->name('analytics.data');

    // Team Management
    Route::prefix('team')->name('team.')->group(function () {
        Route::get('/', [TeamController::class, 'index'])->name('index');
        Route::post('/', [TeamController::class, 'store'])->name('store');
        Route::get('/{id}', [TeamController::class, 'show'])->name('show');
        Route::put('/{id}', [TeamController::class, 'update'])->name('update');
        Route::delete('/{id}', [TeamController::class, 'destroy'])->name('destroy');
        Route::post('/{id}/resend-invitation', [TeamController::class, 'resendInvitation'])->name('resend-invitation');


    });

    // Venues Management
    Route::get('/venues', [BusinessVenueController::class, 'index'])->name('venues.index');
    Route::get('/venues/create', [BusinessVenueController::class, 'create'])->name('venues.create');
    Route::post('/venues', [BusinessVenueController::class, 'store'])->name('venues.store');
    Route::get('/venues/{id}', [BusinessVenueController::class, 'show'])->name('venues.show');
    Route::get('/venues/{id}/edit', [BusinessVenueController::class, 'edit'])->name('venues.edit');
    Route::put('/venues/{id}', [BusinessVenueController::class, 'update'])->name('venues.update');
    Route::delete('/venues/{id}', [BusinessVenueController::class, 'destroy'])->name('venues.destroy');

    // Offers Management
    Route::get('/offers', [BusinessOfferController::class, 'index'])->name('offers.index');
    Route::get('/offers/create', [BusinessOfferController::class, 'create'])->name('offers.create');
    Route::post('/offers', [BusinessOfferController::class, 'store'])->name('offers.store');
    Route::get('/offers/{id}', [BusinessOfferController::class, 'show'])->name('offers.show');
    Route::get('/offers/{id}/edit', [BusinessOfferController::class, 'edit'])->name('offers.edit');
    Route::put('/offers/{id}', [BusinessOfferController::class, 'update'])->name('offers.update');
    Route::delete('/offers/{id}', [BusinessOfferController::class, 'destroy'])->name('offers.destroy');
    Route::post('/offers/{id}/toggle-status', [BusinessOfferController::class, 'toggleStatus'])->name('offers.toggle-status');
    Route::post('/offers/{id}/duplicate', [BusinessOfferController::class, 'duplicate'])->name('offers.duplicate');

    // Redemptions
    Route::get('/redemptions', [BusinessRedemptionController::class, 'index'])->name('redemptions');
    Route::get('/redemptions/export', [BusinessRedemptionController::class, 'export'])->name('redemptions.export');
    Route::get('/redemptions/{id}', [BusinessRedemptionController::class, 'show'])->name('redemptions.show');
    Route::post('/redemptions/{id}/verify', [BusinessRedemptionController::class, 'verify'])->name('redemptions.verify');
    Route::post('/redemptions/{id}/reject', [BusinessRedemptionController::class, 'reject'])->name('redemptions.reject');

    // Subscription
    Route::get('/subscription', [BusinessSubscriptionController::class, 'index'])->name('subscription.index');
    Route::get('/subscription/plans', [BusinessSubscriptionController::class, 'plans'])->name('subscription.plans');
    Route::get('/subscription/checkout/{subscription}', [BusinessSubscriptionController::class, 'checkout'])->name('subscription.checkout');
    Route::post('/subscription/upgrade', [BusinessSubscriptionController::class, 'upgrade'])->name('subscription.upgrade');
    Route::get('/subscription/upgrade', [BusinessSubscriptionController::class, 'plans'])->name('subscription.upgrade.view');
    Route::post('/subscription/cancel', [BusinessSubscriptionController::class, 'cancel'])->name('subscription.cancel');
    Route::post('/subscription/reactivate', [BusinessSubscriptionController::class, 'reactivate'])->name('subscription.reactivate');
    Route::get('/subscription/invoice/{id}', [BusinessSubscriptionController::class, 'showInvoice'])->name('subscription.invoice');
    Route::get('/subscription/invoice/{id}/download', [BusinessSubscriptionController::class, 'downloadInvoice'])->name('subscription.invoice.download');

    // Marketing
    Route::get('/marketing', [BusinessMarketingController::class, 'index'])->name('marketing.index');
    Route::get('/marketing/campaigns', [BusinessMarketingController::class, 'campaigns'])->name('marketing.campaigns');
    Route::post('/marketing/campaigns', [BusinessMarketingController::class, 'createCampaign'])->name('marketing.campaigns.create');

    // Notifications
    Route::get('/notifications', [BusinessNotificationController::class, 'index'])->name('notifications.index');
    Route::get('/notifications/preferences', [BusinessSettingsController::class, 'notificationPreferences'])->name('notifications.preferences');
    Route::post('/notifications/send', [BusinessNotificationController::class, 'send'])->name('notifications.send');
    Route::post('/notifications/{id}/mark-read', [BusinessNotificationController::class, 'markAsRead'])->name('notifications.mark-read');
    Route::post('/notifications/mark-all-read', [BusinessNotificationController::class, 'markAllAsRead'])->name('notifications.mark-all-read');
    Route::delete('/notifications/{id}', [BusinessNotificationController::class, 'destroy'])->name('notifications.destroy');

    // Referral Program
    Route::get('/referral', [BusinessReferralController::class, 'index'])->name('referral.index');
    Route::get('/referral/stats', [BusinessReferralController::class, 'stats'])->name('referral.stats');

    // Support & Tickets
    Route::get('/support', [\App\Http\Controllers\Business\TicketController::class, 'index'])->name('support.index');
    Route::get('/support/tickets', [\App\Http\Controllers\Business\TicketController::class, 'index'])->name('support.tickets');
    Route::get('/support/tickets/create', [\App\Http\Controllers\Business\TicketController::class, 'create'])->name('support.create');
    Route::post('/support/tickets', [\App\Http\Controllers\Business\TicketController::class, 'store'])->name('support.tickets.store');
    Route::get('/support/tickets/{id}', [\App\Http\Controllers\Business\TicketController::class, 'show'])->name('support.tickets.show');
    Route::post('/support/tickets/{id}/reply', [\App\Http\Controllers\Business\TicketController::class, 'reply'])->name('support.tickets.reply');
    Route::post('/support/tickets/{id}/close', [\App\Http\Controllers\Business\TicketController::class, 'close'])->name('support.tickets.close');
    Route::post('/support/tickets/{id}/reopen', [\App\Http\Controllers\Business\TicketController::class, 'reopen'])->name('support.tickets.reopen');

    // Settings
    Route::get('/settings', [BusinessSettingsController::class, 'index'])->name('settings');
    Route::get('/settings/index', [BusinessSettingsController::class, 'index'])->name('settings.index');
    Route::post('/settings/update', [BusinessSettingsController::class, 'update'])->name('settings.update');
    Route::post('/settings/password', [BusinessSettingsController::class, 'updatePassword'])->name('settings.password');
    Route::post('/settings/change-password', [BusinessSettingsController::class, 'updatePassword'])->name('settings.change-password');
    Route::put('/settings/update-profile', [BusinessSettingsController::class, 'updateProfile'])->name('settings.update-profile');
    Route::put('/settings/update-account', [BusinessSettingsController::class, 'updateAccount'])->name('settings.update-account');
    Route::put('/settings/update-preferences', [BusinessSettingsController::class, 'updatePreferences'])->name('settings.update-preferences');
    Route::put('/settings/update-notifications', [BusinessSettingsController::class, 'updateNotifications'])->name('settings.update-notifications');

    // Additional Notification Routes
    Route::post('/notifications/preferences', [BusinessNotificationController::class, 'updatePreferences'])->name('notifications.preferences');

    // Additional Marketing Routes
    Route::get('/marketing/create', [BusinessMarketingController::class, 'create'])->name('marketing.create');
    Route::get('/marketing/{id}/view', [BusinessMarketingController::class, 'view'])->name('marketing.view');

    // Additional Redemption Routes
    Route::get('/redemptions/manual', [BusinessRedemptionController::class, 'manual'])->name('redemptions.manual');

    // Additional Subscription Routes
    Route::post('/subscription/update-payment', [BusinessSubscriptionController::class, 'updatePayment'])->name('subscription.update-payment');

    // Profile Route
    Route::get('/profile', function() {
        return redirect()->route('business.settings');
    })->name('profile');

    // Team Role Management
    Route::post('/team/roles', [TeamController::class, 'storeRole'])->name('roles.store');
});

// Admin Authentication Routes (Public)
Route::prefix('admin')->name('admin.')->group(function () {
    Route::get('/login', [AdminAuthController::class, 'showLoginForm'])->name('login');
    Route::post('/login', [AdminAuthController::class, 'login'])->name('login.submit');
    Route::post('/logout', [AdminAuthController::class, 'logout'])->name('logout');
});

// Admin Panel Routes (Protected)
Route::prefix('admin')->name('admin.')->middleware(['auth'])->group(function () {
    // Dashboard
    Route::get('/dashboard', [AdminDashboardController::class, 'index'])->name('dashboard');
    Route::get('/analytics', [AdminDashboardController::class, 'analytics'])->name('analytics');

    // User Management
    Route::get('/users', [AdminUserController::class, 'index'])->name('users.index');
    Route::post('/users', [AdminUserController::class, 'store'])->name('users.store');
    Route::get('/users/{id}', [AdminUserController::class, 'show'])->name('users.show');
    Route::get('/users/{id}/edit', [AdminUserController::class, 'edit'])->name('users.edit');
    Route::put('/users/{id}', [AdminUserController::class, 'update'])->name('users.update');
    Route::delete('/users/{id}', [AdminUserController::class, 'destroy'])->name('users.destroy');
    Route::post('/users/export', [AdminUserController::class, 'export'])->name('users.export');

    // Business Management
    Route::get('/businesses', [AdminBusinessController::class, 'index'])->name('businesses.index');
    Route::get('/businesses/{id}', [AdminBusinessController::class, 'show'])->name('businesses.show');
    Route::get('/businesses/{id}/verify', [AdminBusinessController::class, 'showVerify'])->name('businesses.verify');
    Route::post('/businesses/{id}/verify', [AdminBusinessController::class, 'verify'])->name('businesses.verify.submit');
    Route::post('/businesses/{id}/approve', [AdminBusinessController::class, 'approve'])->name('businesses.approve');
    Route::post('/businesses/{id}/reject', [AdminBusinessController::class, 'reject'])->name('businesses.reject');
    Route::post('/businesses/{id}/suspend', [AdminBusinessController::class, 'suspend'])->name('businesses.suspend');
    Route::post('/businesses/{id}/activate', [AdminBusinessController::class, 'activate'])->name('businesses.activate');
    Route::post('/businesses/{id}/assign-subscription', [AdminBusinessController::class, 'assignSubscription'])->name('businesses.assign-subscription');
    Route::delete('/businesses/{id}', [AdminBusinessController::class, 'destroy'])->name('businesses.destroy');
    Route::post('/businesses/export', [AdminBusinessController::class, 'export'])->name('businesses.export');

    // Category Management
    Route::get('/categories', [AdminCategoryController::class, 'index'])->name('categories.index');
    Route::get('/categories/{id}', [AdminCategoryController::class, 'show'])->name('categories.show');
    Route::post('/categories', [AdminCategoryController::class, 'store'])->name('categories.store');
    Route::put('/categories/{id}', [AdminCategoryController::class, 'update'])->name('categories.update');
    Route::delete('/categories/{id}', [AdminCategoryController::class, 'destroy'])->name('categories.destroy');
    Route::post('/categories/{id}/toggle', [AdminCategoryController::class, 'toggle'])->name('categories.toggle');

    // Spotlight Management
    Route::get('/spotlights', [AdminSpotlightController::class, 'index'])->name('spotlights.index');
    Route::get('/spotlights/{id}', [AdminSpotlightController::class, 'show'])->name('spotlights.show');
    Route::post('/spotlights/{id}/approve', [AdminSpotlightController::class, 'approve'])->name('spotlights.approve');
    Route::post('/spotlights/{id}/reject', [AdminSpotlightController::class, 'reject'])->name('spotlights.reject');
    Route::post('/spotlights/{id}/unflag', [AdminSpotlightController::class, 'unflag'])->name('spotlights.unflag');
    Route::delete('/spotlights/{id}', [AdminSpotlightController::class, 'destroy'])->name('spotlights.destroy');

    // Subscription Plans (Business & User Plans)
    Route::get('/subscription-plans', [AdminSubscriptionPlanController::class, 'index'])->name('subscription-plans.index');
    Route::get('/subscription-plans/create', [AdminSubscriptionPlanController::class, 'create'])->name('subscription-plans.create');
    Route::post('/subscription-plans', [AdminSubscriptionPlanController::class, 'store'])->name('subscription-plans.store');
    Route::get('/subscription-plans/{id}/edit', [AdminSubscriptionPlanController::class, 'edit'])->name('subscription-plans.edit');
    Route::put('/subscription-plans/{id}', [AdminSubscriptionPlanController::class, 'update'])->name('subscription-plans.update');
    Route::delete('/subscription-plans/{id}', [AdminSubscriptionPlanController::class, 'destroy'])->name('subscription-plans.destroy');
    Route::post('/subscription-plans/{id}/toggle', [AdminSubscriptionPlanController::class, 'toggle'])->name('subscription-plans.toggle');
    Route::post('/subscription-plans/{id}/toggle-popular', [AdminSubscriptionPlanController::class, 'togglePopular'])->name('subscription-plans.toggle-popular');

    // User Subscriptions Management
    Route::get('/user-subscriptions', [\App\Http\Controllers\Admin\UserSubscriptionController::class, 'index'])->name('user-subscriptions.index');
    Route::get('/user-subscriptions/{id}', [\App\Http\Controllers\Admin\UserSubscriptionController::class, 'show'])->name('user-subscriptions.show');
    Route::post('/user-subscriptions/{id}/cancel', [\App\Http\Controllers\Admin\UserSubscriptionController::class, 'cancel'])->name('user-subscriptions.cancel');
    Route::post('/user-subscriptions/{id}/activate', [\App\Http\Controllers\Admin\UserSubscriptionController::class, 'activate'])->name('user-subscriptions.activate');
    Route::get('/user-subscriptions/analytics', [\App\Http\Controllers\Admin\UserSubscriptionController::class, 'analytics'])->name('user-subscriptions.analytics');
    Route::get('/subscription-plans/{id}/subscribers', [AdminSubscriptionPlanController::class, 'subscribers'])->name('subscription-plans.subscribers');

    // Venue Management
    Route::get('/venues', [AdminVenueController::class, 'index'])->name('venues.index');
    Route::get('/venues/export', [AdminVenueController::class, 'export'])->name('venues.export');
    Route::get('/venues/{id}', [AdminVenueController::class, 'show'])->name('venues.show');
    Route::get('/venues/{id}/edit', [AdminVenueController::class, 'edit'])->name('venues.edit');
    Route::put('/venues/{id}', [AdminVenueController::class, 'update'])->name('venues.update');
    Route::post('/venues/{id}/approve', [AdminVenueController::class, 'approve'])->name('venues.approve');
    Route::post('/venues/{id}/reject', [AdminVenueController::class, 'reject'])->name('venues.reject');
    Route::post('/venues/{id}/toggle-status', [AdminVenueController::class, 'toggleStatus'])->name('venues.toggle-status');
    Route::post('/venues/{id}/toggle-featured', [AdminVenueController::class, 'toggleFeatured'])->name('venues.toggle-featured');
    Route::post('/venues/{id}/update-status', [AdminVenueController::class, 'updateStatus'])->name('venues.update-status');
    Route::delete('/venues/{id}', [AdminVenueController::class, 'destroy'])->name('venues.destroy');

    // Offer Management
    Route::get('/offers', [AdminOfferController::class, 'index'])->name('offers.index');
    Route::get('/offers/{id}', [AdminOfferController::class, 'show'])->name('offers.show');
    Route::get('/offers/{id}/edit', [AdminOfferController::class, 'edit'])->name('offers.edit');
    Route::put('/offers/{id}', [AdminOfferController::class, 'update'])->name('offers.update');
    Route::post('/offers/{id}/approve', [AdminOfferController::class, 'approve'])->name('offers.approve');
    Route::post('/offers/{id}/reject', [AdminOfferController::class, 'reject'])->name('offers.reject');
    Route::post('/offers/{id}/pause', [AdminOfferController::class, 'pause'])->name('offers.pause');
    Route::post('/offers/{id}/suspend', [AdminOfferController::class, 'suspend'])->name('offers.suspend');
    Route::post('/offers/{id}/activate', [AdminOfferController::class, 'activate'])->name('offers.activate');
    Route::delete('/offers/{id}', [AdminOfferController::class, 'destroy'])->name('offers.destroy');

    // Audit Logs
    Route::get('/audit-logs', [AdminAuditController::class, 'index'])->name('audit-logs.index');
    Route::get('/audit-logs/{id}', [AdminAuditController::class, 'show'])->name('audit-logs.show');
    Route::post('/audit-logs/export', [AdminAuditController::class, 'export'])->name('audit-logs.export');

    // System Configuration
    Route::get('/configuration', [AdminConfigurationController::class, 'index'])->name('configuration.index');
    Route::post('/configuration/update', [AdminConfigurationController::class, 'update'])->name('configuration.update');

    // FAQ Management
    Route::get('/faqs', [AdminFaqController::class, 'index'])->name('faqs.index');
    Route::post('/faqs', [AdminFaqController::class, 'store'])->name('faqs.store');
    Route::put('/faqs/{id}', [AdminFaqController::class, 'update'])->name('faqs.update');
    Route::delete('/faqs/{id}', [AdminFaqController::class, 'destroy'])->name('faqs.destroy');
    Route::post('/faqs/{id}/toggle', [AdminFaqController::class, 'toggle'])->name('faqs.toggle');

    // GDPR Management
    Route::get('/gdpr', [AdminGdprController::class, 'index'])->name('gdpr.index');
    Route::get('/gdpr/{id}', [AdminGdprController::class, 'show'])->name('gdpr.show');
    Route::post('/gdpr/{id}/approve', [AdminGdprController::class, 'approve'])->name('gdpr.approve');
    Route::post('/gdpr/{id}/reject', [AdminGdprController::class, 'reject'])->name('gdpr.reject');
    Route::post('/gdpr/{id}/complete', [AdminGdprController::class, 'complete'])->name('gdpr.complete');

    // Marketing Management
    Route::get('/marketing', [AdminMarketingController::class, 'index'])->name('marketing.index');
    Route::get('/marketing/campaigns', [AdminMarketingController::class, 'campaigns'])->name('marketing.campaigns');
    Route::post('/marketing/campaigns', [AdminMarketingController::class, 'createCampaign'])->name('marketing.campaigns.create');
    Route::get('/marketing/analytics', [AdminMarketingController::class, 'analytics'])->name('marketing.analytics');

    // Roles Management
    Route::get('/roles', [\App\Http\Controllers\Admin\RoleController::class, 'index'])->name('roles.index');
    Route::get('/roles/create', [\App\Http\Controllers\Admin\RoleController::class, 'create'])->name('roles.create');
    Route::post('/roles', [\App\Http\Controllers\Admin\RoleController::class, 'store'])->name('roles.store');
    Route::get('/roles/{id}', [\App\Http\Controllers\Admin\RoleController::class, 'show'])->name('roles.show');
    Route::get('/roles/{id}/edit', [\App\Http\Controllers\Admin\RoleController::class, 'edit'])->name('roles.edit');
    Route::put('/roles/{id}', [\App\Http\Controllers\Admin\RoleController::class, 'update'])->name('roles.update');
    Route::delete('/roles/{id}', [\App\Http\Controllers\Admin\RoleController::class, 'destroy'])->name('roles.destroy');
    Route::get('/roles/{id}/permissions', [\App\Http\Controllers\Admin\RoleController::class, 'permissions'])->name('roles.permissions');
    Route::post('/roles/{id}/permissions', [\App\Http\Controllers\Admin\RoleController::class, 'updatePermissions'])->name('roles.permissions.update');
    Route::get('/roles/{id}/users', [\App\Http\Controllers\Admin\RoleController::class, 'users'])->name('roles.users');

    // Permissions Management
    Route::get('/permissions', [\App\Http\Controllers\Admin\PermissionController::class, 'index'])->name('permissions.index');
    Route::get('/permissions/create', [\App\Http\Controllers\Admin\PermissionController::class, 'create'])->name('permissions.create');
    Route::post('/permissions', [\App\Http\Controllers\Admin\PermissionController::class, 'store'])->name('permissions.store');
    Route::get('/permissions/{id}', [\App\Http\Controllers\Admin\PermissionController::class, 'show'])->name('permissions.show');
    Route::get('/permissions/{id}/edit', [\App\Http\Controllers\Admin\PermissionController::class, 'edit'])->name('permissions.edit');
    Route::put('/permissions/{id}', [\App\Http\Controllers\Admin\PermissionController::class, 'update'])->name('permissions.update');
    Route::delete('/permissions/{id}', [\App\Http\Controllers\Admin\PermissionController::class, 'destroy'])->name('permissions.destroy');
    Route::get('/permissions/{id}/roles', [\App\Http\Controllers\Admin\PermissionController::class, 'roles'])->name('permissions.roles');
    Route::post('/permissions/bulk-assign', [\App\Http\Controllers\Admin\PermissionController::class, 'bulkAssign'])->name('permissions.bulk-assign');

    // Reviews Management
    Route::get('/reviews', function() {
        $reviews = \App\Models\Review::with(['user', 'venue'])->latest()->paginate(20);
        return view('admin.reviews.index', compact('reviews'));
    })->name('reviews.index');

    // Payments Management
    Route::get('/payments', function() {
        $payments = \App\Models\Payment::with(['business'])->latest()->paginate(20);
        return view('admin.payments.index', compact('payments'));
    })->name('payments.index');
    Route::get('/payments/{id}', function($id) {
        $payment = \App\Models\Payment::with(['business', 'subscription.plan'])->findOrFail($id);
        return response()->json([
            'success' => true,
            'payment' => $payment
        ]);
    })->name('payments.show');

    // Subscriptions Management
    Route::get('/subscriptions', function() {
        $subscriptions = \App\Models\Subscription::with(['business', 'plan'])->latest()->paginate(20);
        return view('admin.subscriptions.index', compact('subscriptions'));
    })->name('subscriptions.index');
    Route::get('/subscriptions/{id}', function($id) {
        $subscription = \App\Models\Subscription::with(['business', 'plan'])->findOrFail($id);
        return response()->json([
            'success' => true,
            'subscription' => $subscription
        ]);
    })->name('subscriptions.show');

    // Support Tickets Management
    Route::get('/tickets', function() {
        $tickets = \App\Models\SupportTicket::with(['user'])->latest()->paginate(20);
        return view('admin.tickets.index', compact('tickets'));
    })->name('tickets.index');

    // Moderation Dashboard
    Route::get('/moderation', function() {
        $pendingSpotlights = \App\Models\Spotlight::where('moderation_status', 'pending')->count();
        $flaggedContent = \App\Models\Spotlight::where('status', 'flagged')->count();
        $pendingReviews = \App\Models\Review::where('status', 'pending')->count();
        return view('admin.moderation.index', compact('pendingSpotlights', 'flaggedContent', 'pendingReviews'));
    })->name('moderation.index');

    // Reports & Analytics
    Route::get('/reports', function() {
        return view('admin.reports.index');
    })->name('reports.index');

    // Notifications Management
    Route::get('/notifications', function() {
        $notifications = auth()->user()->notifications()->paginate(20);
        return view('admin.notifications.index', compact('notifications'));
    })->name('notifications.index');

    // Alias for spotlight (singular)
    Route::get('/spotlight', [AdminSpotlightController::class, 'index'])->name('spotlight.index');

    // Settings
    Route::get('/settings', [AdminSettingsController::class, 'index'])->name('settings.index');
    Route::post('/settings/update/{section}', [AdminSettingsController::class, 'update'])->name('settings.update');
    Route::post('/settings/maintenance', [AdminSettingsController::class, 'maintenance'])->name('settings.maintenance');
    Route::post('/settings/test-email', [AdminSettingsController::class, 'testEmail'])->name('settings.test-email');
    Route::post('/settings/test-sms', [AdminSettingsController::class, 'testSMS'])->name('settings.test-sms');
});

