<?php

namespace App\Helpers;

use App\Models\MarketingCampaign;
use Illuminate\Support\Facades\Auth;

class AdminHelper
{
    /**
     * Check if current user is admin
     */
    public static function isAdmin()
    {
        if (!Auth::check()) {
            return false;
        }

        return Auth::user()->hasAnyRole(['admin', 'super_admin']) ||
               Auth::user()->hasPermissionTo('admin.access');
    }

    /**
     * Check if current user is super admin
     */
    public static function isSuperAdmin()
    {
        if (!Auth::check()) {
            return false;
        }

        return Auth::user()->hasRole('super_admin');
    }

    /**
     * Check if current user has permission
     */
    public static function can($permission)
    {
        if (!Auth::check()) {
            return false;
        }

        $user = Auth::user();

        // Super admin and admin roles have all permissions by default
        if ($user->hasAnyRole(['super_admin', 'admin'])) {
            return true;
        }

        // Otherwise check for specific permission
        return $user->hasPermissionTo($permission);
    }

    /**
     * Check if current user has any of the permissions
     */
    public static function canAny($permissions)
    {
        if (!Auth::check()) {
            return false;
        }

        $user = Auth::user();

        // Super admin and admin roles have all permissions by default
        if ($user->hasAnyRole(['super_admin', 'admin'])) {
            return true;
        }

        foreach ($permissions as $permission) {
            if ($user->hasPermissionTo($permission)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if current user has all permissions
     */
    public static function canAll($permissions)
    {
        if (!Auth::check()) {
            return false;
        }

        $user = Auth::user();

        // Super admin and admin roles have all permissions by default
        if ($user->hasAnyRole(['super_admin', 'admin'])) {
            return true;
        }

        foreach ($permissions as $permission) {
            if (!$user->hasPermissionTo($permission)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get all admin menu items based on permissions
     */
    public static function getAdminMenuItems()
    {
        if (!static::isAdmin()) {
            return [];
        }

        $user = Auth::user();
        $items = [];

        // For super_admin and admin, show all items (they have full access)
        // For others, check specific permissions
        $isSuperAdmin = $user->hasRole('super_admin');
        $isAdmin = $user->hasRole('admin');
        $showAll = $isSuperAdmin || $isAdmin;

        // Dashboard
        if ($showAll || static::can('admin.dashboard')) {
            $items[] = [
                'title' => 'Dashboard',
                'url' => route('admin.dashboard'),
                'icon' => 'fas fa-tachometer-alt',
                'active' => request()->routeIs('admin.dashboard*'),
            ];
        }

        // Users Management
        if ($showAll || static::can('users.view')) {
            $items[] = [
                'title' => 'Users',
                'url' => route('admin.users.index'),
                'icon' => 'fas fa-users',
                'active' => request()->routeIs('admin.users*'),
                'badge' => ($showAll || static::can('users.create')) ? 'Can Create' : null,
            ];
        }

        // User Activity
        if ($showAll || static::can('users.view')) {
            $items[] = [
                'title' => 'User Activity',
                'url' => route('admin.users.activity'),
                'icon' => 'fas fa-history',
                'active' => request()->routeIs('admin.users.activity'),
            ];
        }

        // Roles & Permissions
        if ($showAll || static::canAny(['roles.view', 'permissions.view'])) {
            $items[] = [
                'title' => 'Roles & Permissions',
                'icon' => 'fas fa-shield-alt',
                'active' => request()->routeIs('admin.roles*') || request()->routeIs('admin.permissions*'),
                'submenu' => array_filter([
                    ($showAll || static::can('roles.view')) ? [
                        'title' => 'Roles',
                        'url' => route('admin.roles.index'),
                        'active' => request()->routeIs('admin.roles*'),
                    ] : null,
                    ($showAll || static::can('permissions.view')) ? [
                        'title' => 'Permissions',
                        'url' => route('admin.permissions.index'),
                        'active' => request()->routeIs('admin.permissions*'),
                    ] : null,
                ]),
            ];
        }

        // Activity & Audit
        if ($showAll || static::can('activities.view')) {
            $items[] = [
                'title' => 'Activity Logs',
                'url' => route('admin.activities.index'),
                'icon' => 'fas fa-history',
                'active' => request()->routeIs('admin.activities*'),
            ];
        }

        // Advanced Analytics
        if ($showAll || static::can('activities.view')) {
            $items[] = [
                'title' => 'Advanced Analytics',
                'url' => route('admin.analytics.index'),
                'icon' => 'fas fa-chart-bar',
                'active' => request()->routeIs('admin.analytics*'),
            ];
        }

        // Sessions
        if ($showAll || static::can('sessions.view')) {
            $items[] = [
                'title' => 'Admin Sessions',
                'url' => route('admin.sessions.index'),
                'icon' => 'fas fa-users-cog',
                'active' => request()->routeIs('admin.sessions*'),
            ];
        }

//        Business
        if ($showAll || static::canAny(['businesses.view', 'businesses.create'])) {
            $items[] = [
                'title' => 'Businesses',
                'url' => route('admin.businesses.index'),
                'icon' => 'fas fa-briefcase',
                'active' => request()->routeIs('admin.businesses*'),
            ];
        }

//        Venues
        if ($showAll || static::canAny(['venues.view', 'venues.create'])) {
            $items[] = [
                'title' => 'Venues',
                'url' => route('admin.venues.index'),
                'icon' => 'fas fa-building',
                'active' => request()->routeIs('admin.venues*'),
            ];
        }

//        Spotlight
        if ($showAll || static::canAny(['spotlight.view', 'spotlight.create'])) {
            $items[] = [
                'title' => 'Spotlights',
                'url' => route('admin.spotlight.index'),
                'icon' => 'fas fa-lightbulb',
                'active' => request()->routeIs('admin.spotlight*'),
            ];
        }

        // Teams Management
        if ($showAll || static::canAny(['teams.view', 'teams.create'])) {
            $items[] = [
                'title' => 'Teams',
                'url' => route('admin.teams.index'),
                'icon' => 'fas fa-users-cog',
                'active' => request()->routeIs('admin.teams*'),
            ];
        }

        // Staff Management
        if ($showAll || static::canAny(['staff.view', 'staff.create'])) {
            $items[] = [
                'title' => 'Staff',
                'url' => route('admin.staff.index'),
                'icon' => 'fas fa-user-tie',
                'active' => request()->routeIs('admin.staff*'),
            ];
        }

        // Chats Management
        if ($showAll || static::canAny(['chats.view', 'chats.create'])) {
            $items[] = [
                'title' => 'Chats & Messages',
                'url' => route('admin.chats.index'),
                'icon' => 'fas fa-comments',
                'active' => request()->routeIs('admin.chats*'),
            ];
        }

        // Notification Templates
        if ($showAll || static::can('notifications.manage_templates')) {
            $items[] = [
                'title' => 'Notifications',
                'icon' => 'fas fa-bell',
                'active' => request()->routeIs('admin.notifications*'),
                'submenu' => array_filter([
                    [
                        'title' => 'Email Templates',
                        'url' => route('admin.notifications.email-templates.index'),
                        'active' => request()->routeIs('admin.notifications.email-templates*'),
                    ],
                    [
                        'title' => 'SMS Templates',
                        'url' => route('admin.notifications.sms-templates.index'),
                        'active' => request()->routeIs('admin.notifications.sms-templates*'),
                    ],
                ]),
            ];
        }

        // Categories
        if ($showAll || static::canAny(['categories.view', 'categories.create'])) {
            $items[] = [
                'title' => 'Categories',
                'url' => route('admin.categories.index'),
                'icon' => 'fas fa-list',
                'active' => request()->routeIs('admin.categories*'),
            ];
        }

        // Offers Management
        if ($showAll || static::canAny(['offers.view', 'offers.create'])) {
            $items[] = [
                'title' => 'Offers',
                'url' => route('admin.offers.index'),
                'icon' => 'fas fa-tag',
                'active' => request()->routeIs('admin.offers*'),
            ];
        }

        // Reviews Management
        if ($showAll || static::can('reviews.view')) {
            $items[] = [
                'title' => 'Reviews',
                'url' => route('admin.reviews.index'),
                'icon' => 'fas fa-star',
                'active' => request()->routeIs('admin.reviews*'),
            ];
        }

        // Marketing Campaigns (Banners)
        if ($showAll || static::canAny(['marketing.view', 'marketing.create'])) {
            $items[] = [
                'title' => 'Marketing Campaigns',
                'url' => route('admin.marketing.campaigns'),
                'icon' => 'fas fa-banner',
                'active' => request()->routeIs('admin.marketing*'),
                'badge' => MarketingCampaign::where('status', 'pending')->count() > 0 ? MarketingCampaign::where('status', 'pending')->count() . ' pending' : null,
            ];
        }

        // Subscriptions Management
        if ($showAll || static::canAny(['subscriptions.view', 'subscriptions.manage'])) {
            $items[] = [
                'title' => 'Subscriptions',
                'icon' => 'fas fa-credit-card',
                'active' => request()->routeIs('admin.subscription*') || request()->routeIs('admin.user-subscriptions*'),
                'submenu' => array_filter([
                    [
                        'title' => 'Plans',
                        'url' => route('admin.subscription-plans.index'),
                        'active' => request()->routeIs('admin.subscription-plans*'),
                    ],
                    [
                        'title' => 'User Subscriptions',
                        'url' => route('admin.user-subscriptions.index'),
                        'active' => request()->routeIs('admin.user-subscriptions*'),
                    ],
                ]),
            ];
        }

        // Support Tickets
        if ($showAll || static::can('support.view')) {
            $items[] = [
                'title' => 'Support Tickets',
                'url' => route('admin.tickets.index'),
                'icon' => 'fas fa-ticket-alt',
                'active' => request()->routeIs('admin.tickets*'),
            ];
        }

        // Configuration (Consolidated - includes both Configuration and Settings)
        if ($showAll || static::canAny(['configurations.view', 'settings.view'])) {
            $items[] = [
                'title' => 'Configuration & Settings',
                'url' => route('admin.configurations.index'),
                'icon' => 'fas fa-cog',
                'active' => request()->routeIs('admin.configurations*') || request()->routeIs('admin.settings*'),
            ];
        }


        return $items;
    }

    /**
     * Check if section should be visible
     */
    public static function canViewSection($section)
    {
        $sectionPermissions = [
            'dashboard' => 'admin.dashboard',
            'users' => 'users.view',
            'roles' => 'roles.view',
            'permissions' => 'permissions.view',
            'activities' => 'activities.view',
            'sessions' => 'sessions.view',
            'configurations' => 'configurations.view',
            'settings' => 'settings.view',
            'audit' => 'activities.view',
        ];

        $permission = $sectionPermissions[$section] ?? null;

        if (!$permission) {
            return false;
        }

        return static::can($permission);
    }

    /**
     * Get admin statistics
     */
    public static function getStats()
    {
        if (!static::isAdmin()) {
            return [];
        }

        return [
            'users' => \App\Models\User::count(),
            'active_users' => \App\Models\User::where('status', 'active')->count(),
            'sessions' => \App\Models\AdminSession::active()->count(),
            'activities_today' => \App\Models\AdminActivityLog::whereDate('created_at', today())->count(),
        ];
    }
}

