<?php

namespace App\Http\Controllers\API\V1\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class LoginController extends Controller
{
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|string',
            'device_token' => 'nullable|string',
            'fcm_token' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        if ($user->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Account is not active. Please verify your account or contact support.',
                'status' => $user->status
            ], 403);
        }

        // Update device tokens
        if ($request->device_token) {
            $user->device_token = $request->device_token;
        }
        if ($request->fcm_token) {
            $user->fcm_token = $request->fcm_token;
        }
        $user->last_active_at = now();
        $user->save();

        $token = $user->createToken('auth_token')->plainTextToken;

        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                    'full_name' => $user->full_name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'profile_image' => $user->profile_image,
                    'account_type' => $user->account_type,
                    'language' => $user->language,
                    'dark_mode' => $user->dark_mode,
                    'has_active_subscription' => $user->hasActiveSubscription(),
                ],
                'access_token' => $token,
                'token_type' => 'Bearer',
            ]
        ]);
    }

    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Logged out successfully'
        ]);
    }

    public function me(Request $request)
    {
        $user = $request->user();

        return response()->json([
            'success' => true,
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                    'full_name' => $user->full_name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'profile_image' => $user->profile_image,
                    'gender' => $user->gender,
                    'date_of_birth' => $user->date_of_birth,
                    'account_type' => $user->account_type,
                    'status' => $user->status,
                    'language' => $user->language,
                    'dark_mode' => $user->dark_mode,
                    'interests' => $user->interests,
                    'preferences' => $user->preferences,
                    'referral_code' => $user->referral_code,
                    'has_active_subscription' => $user->hasActiveSubscription(),
                    'created_at' => $user->created_at,
                ]
            ]
        ]);
    }
}
