<?php

namespace App\Http\Controllers\API\V1\Business;

use App\Http\Controllers\Controller;
use App\Models\Venue;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class VenueManagementController extends Controller
{
    /**
     * Get all venues for authenticated business
     */
    public function index(Request $request)
    {
        $business = auth()->user()->business;

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'You are not associated with any business'
            ], 403);
        }

        $query = Venue::where('business_id', $business->id);

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_ar', 'like', "%{$search}%")
                  ->orWhere('address', 'like', "%{$search}%");
            });
        }

        $venues = $query->orderBy('created_at', 'desc')
            ->paginate($request->per_page ?? 15);

        return response()->json([
            'success' => true,
            'data' => $venues
        ]);
    }

    /**
     * Create new venue
     */
    public function store(Request $request)
    {
        $business = auth()->user()->business;

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'You are not associated with any business'
            ], 403);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'address' => 'required|string',
            'address_ar' => 'nullable|string',
            'city' => 'required|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'required|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'latitude' => 'required|numeric|between:-90,90',
            'longitude' => 'required|numeric|between:-180,180',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'amenities' => 'nullable|array',
            'amenities.*' => 'string',
            'working_hours' => 'nullable|array',
            'images' => 'nullable|array',
            'images.*' => 'image|mimes:jpeg,png,jpg|max:2048',
            'status' => 'sometimes|in:active,inactive',
        ]);

        // Handle image uploads
        $images = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $path = $image->store('venues', 'public');
                $images[] = $path;
            }
        }

        $venue = Venue::create([
            'business_id' => $business->id,
            'name' => $validated['name'],
            'name_ar' => $validated['name_ar'] ?? null,
            'address' => $validated['address'],
            'address_ar' => $validated['address_ar'] ?? null,
            'city' => $validated['city'],
            'state' => $validated['state'] ?? null,
            'country' => $validated['country'],
            'postal_code' => $validated['postal_code'] ?? null,
            'latitude' => $validated['latitude'],
            'longitude' => $validated['longitude'],
            'phone' => $validated['phone'],
            'email' => $validated['email'] ?? null,
            'description' => $validated['description'] ?? null,
            'description_ar' => $validated['description_ar'] ?? null,
            'amenities' => json_encode($validated['amenities'] ?? []),
            'working_hours' => json_encode($validated['working_hours'] ?? []),
            'images' => json_encode($images),
            'status' => $validated['status'] ?? 'active',
            'rating' => 0,
            'total_reviews' => 0,
            'view_count' => 0,
        ]);

        // Update business total venues count
        $business->increment('total_venues');

        return response()->json([
            'success' => true,
            'message' => 'Venue created successfully',
            'data' => $venue
        ], 201);
    }

    /**
     * Get single venue
     */
    public function show($id)
    {
        $business = auth()->user()->business;

        $venue = Venue::where('business_id', $business->id)
            ->with(['offers', 'reviews'])
            ->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $venue
        ]);
    }

    /**
     * Update venue
     */
    public function update(Request $request, $id)
    {
        $business = auth()->user()->business;

        $venue = Venue::where('business_id', $business->id)->findOrFail($id);

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'address' => 'sometimes|string',
            'address_ar' => 'nullable|string',
            'city' => 'sometimes|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'sometimes|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'latitude' => 'sometimes|numeric|between:-90,90',
            'longitude' => 'sometimes|numeric|between:-180,180',
            'phone' => 'sometimes|string|max:20',
            'email' => 'nullable|email|max:255',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'amenities' => 'nullable|array',
            'working_hours' => 'nullable|array',
            'images' => 'nullable|array',
            'images.*' => 'image|mimes:jpeg,png,jpg|max:2048',
            'status' => 'sometimes|in:active,inactive',
        ]);

        // Handle new image uploads
        if ($request->hasFile('images')) {
            $images = [];
            foreach ($request->file('images') as $image) {
                $path = $image->store('venues', 'public');
                $images[] = $path;
            }
            $validated['images'] = json_encode($images);
        }

        // Convert arrays to JSON
        if (isset($validated['amenities'])) {
            $validated['amenities'] = json_encode($validated['amenities']);
        }
        if (isset($validated['working_hours'])) {
            $validated['working_hours'] = json_encode($validated['working_hours']);
        }

        $venue->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Venue updated successfully',
            'data' => $venue
        ]);
    }

    /**
     * Delete venue
     */
    public function destroy($id)
    {
        $business = auth()->user()->business;

        $venue = Venue::where('business_id', $business->id)->findOrFail($id);

        // Check if venue has active offers
        if ($venue->offers()->where('status', 'active')->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete venue with active offers. Please deactivate or delete offers first.'
            ], 400);
        }

        // Soft delete
        $venue->delete();

        // Update business total venues count
        $business->decrement('total_venues');

        return response()->json([
            'success' => true,
            'message' => 'Venue deleted successfully'
        ]);
    }

    /**
     * Get venue statistics
     */
    public function statistics($id)
    {
        $business = auth()->user()->business;

        $venue = Venue::where('business_id', $business->id)
            ->with(['offers'])
            ->findOrFail($id);

        $stats = [
            'total_offers' => $venue->offers()->count(),
            'active_offers' => $venue->offers()->where('status', 'active')->count(),
            'total_views' => $venue->view_count,
            'total_reviews' => $venue->total_reviews,
            'average_rating' => $venue->rating,
            'total_redemptions' => $venue->offers()->sum('redemption_count'),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }

    /**
     * Get nearby venues
     */
    public function nearby(Request $request)
    {
        $validated = $request->validate([
            'latitude' => 'required|numeric|between:-90,90',
            'longitude' => 'required|numeric|between:-180,180',
            'radius' => 'nullable|integer|min:1|max:50', // in kilometers
        ]);

        $radius = $validated['radius'] ?? 10;
        $lat = $validated['latitude'];
        $lng = $validated['longitude'];

        // Haversine formula to calculate distance
        $venues = Venue::selectRaw("
                *,
                (6371 * acos(cos(radians(?)) * cos(radians(latitude)) *
                cos(radians(longitude) - radians(?)) + sin(radians(?)) *
                sin(radians(latitude)))) AS distance
            ", [$lat, $lng, $lat])
            ->where('status', 'active')
            ->having('distance', '<=', $radius)
            ->orderBy('distance')
            ->with(['business'])
            ->limit(20)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $venues
        ]);
    }
}
