<?php

namespace App\Http\Controllers\API\V1\Spotlight;

use App\Http\Controllers\Controller;
use App\Models\Spotlight;
use App\Models\SpotlightLike;
use App\Models\SpotlightComment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class SpotlightController extends Controller
{
    public function index(Request $request)
    {
        $query = Spotlight::with(['user'])
            ->where('moderation_status', 'approved')
            ->where('status', 'approved');

        // Filter by user
        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // Sort
        $sortBy = $request->sort_by ?? 'created_at';
        $sortOrder = $request->sort_order ?? 'desc';

        if (in_array($sortBy, ['likes_count', 'comments_count', 'view_count', 'created_at'])) {
            $query->orderBy($sortBy, $sortOrder);
        }

        $perPage = $request->per_page ?? 15;
        $spotlights = $query->paginate($perPage);

        // Add is_liked status for authenticated users
        if (auth('sanctum')->check()) {
            $spotlights->getCollection()->transform(function ($spotlight) {
                $spotlight->is_liked = SpotlightLike::where('spotlight_id', $spotlight->id)
                    ->where('user_id', auth()->id())
                    ->exists();
                return $spotlight;
            });
        }

        return response()->json([
            'success' => true,
            'data' => $spotlights
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'content' => 'required|string|max:2000',
            'media_type' => 'nullable|in:image,video,mixed',
            'media_urls' => 'nullable|array',
            'media_urls.*' => 'url',
            'location' => 'nullable|string|max:255',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error',
                'errors' => $validator->errors()
            ], 422);
        }

        $spotlight = Spotlight::create([
            'user_id' => auth()->id(),
            'content' => $request->content,
            'media_type' => $request->media_type,
            'media_urls' => $request->media_urls,
            'location' => $request->location,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'status' => 'pending',
            'moderation_status' => 'pending',
        ]);

        // TODO: Queue for AI moderation
        // TODO: Notify moderators

        return response()->json([
            'success' => true,
            'message' => 'Spotlight created successfully and pending moderation',
            'data' => $spotlight
        ], 201);
    }

    public function show($id)
    {
        $spotlight = Spotlight::with([
            'user',
            'comments' => function($query) {
                $query->where('status', 'active')
                    ->whereNull('parent_id')
                    ->with(['user', 'replies.user'])
                    ->orderBy('created_at', 'desc')
                    ->take(20);
            }
        ])->findOrFail($id);

        // Check moderation status
        if ($spotlight->moderation_status !== 'approved' && $spotlight->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Spotlight not found or pending moderation'
            ], 404);
        }

        // Increment view count
        $spotlight->incrementViews();

        // Check if liked by current user
        $isLiked = false;
        if (auth('sanctum')->check()) {
            $isLiked = SpotlightLike::where('spotlight_id', $spotlight->id)
                ->where('user_id', auth()->id())
                ->exists();
        }

        $spotlight->is_liked = $isLiked;

        return response()->json([
            'success' => true,
            'data' => $spotlight
        ]);
    }

    public function update(Request $request, $id)
    {
        $spotlight = Spotlight::where('user_id', auth()->id())->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'content' => 'sometimes|required|string|max:2000',
            'location' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error',
                'errors' => $validator->errors()
            ], 422);
        }

        $spotlight->update($request->only(['content', 'location']));

        return response()->json([
            'success' => true,
            'message' => 'Spotlight updated successfully',
            'data' => $spotlight
        ]);
    }

    public function destroy($id)
    {
        $spotlight = Spotlight::where('user_id', auth()->id())->findOrFail($id);
        $spotlight->delete();

        return response()->json([
            'success' => true,
            'message' => 'Spotlight deleted successfully'
        ]);
    }

    public function like($id)
    {
        $spotlight = Spotlight::findOrFail($id);

        $like = SpotlightLike::where('spotlight_id', $spotlight->id)
            ->where('user_id', auth()->id())
            ->first();

        if ($like) {
            $like->delete();
            $spotlight->decrementLikes();
            $message = 'Spotlight unliked';
            $isLiked = false;
        } else {
            SpotlightLike::create([
                'spotlight_id' => $spotlight->id,
                'user_id' => auth()->id(),
            ]);
            $spotlight->incrementLikes();
            $message = 'Spotlight liked';
            $isLiked = true;

            // TODO: Notify spotlight owner
        }

        return response()->json([
            'success' => true,
            'message' => $message,
            'data' => [
                'is_liked' => $isLiked,
                'likes_count' => $spotlight->fresh()->likes_count
            ]
        ]);
    }

    public function comment(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'comment' => 'required|string|max:500',
            'parent_id' => 'nullable|exists:spotlight_comments,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error',
                'errors' => $validator->errors()
            ], 422);
        }

        $spotlight = Spotlight::findOrFail($id);

        $comment = SpotlightComment::create([
            'spotlight_id' => $spotlight->id,
            'user_id' => auth()->id(),
            'parent_id' => $request->parent_id,
            'comment' => $request->comment,
            'status' => 'active',
        ]);

        $spotlight->incrementComments();

        // TODO: Notify spotlight owner
        // TODO: If reply, notify parent comment author

        return response()->json([
            'success' => true,
            'message' => 'Comment added successfully',
            'data' => $comment->load('user')
        ], 201);
    }

    public function flag(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'reason' => 'required|in:spam,inappropriate,harassment,other',
            'description' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error',
                'errors' => $validator->errors()
            ], 422);
        }

        $spotlight = Spotlight::findOrFail($id);

        // Check if already flagged by this user
        // TODO: Create SpotlightFlag model and check

        // TODO: Create flag record
        // TODO: Notify moderators if threshold reached

        $spotlight->update(['status' => 'flagged']);

        return response()->json([
            'success' => true,
            'message' => 'Spotlight flagged for review. Thank you for helping keep our community safe.'
        ]);
    }

    public function mySpotlights(Request $request)
    {
        $spotlights = Spotlight::with(['user'])
            ->where('user_id', auth()->id())
            ->orderBy('created_at', 'desc')
            ->paginate($request->per_page ?? 15);

        return response()->json([
            'success' => true,
            'data' => $spotlights
        ]);
    }

    public function trending(Request $request)
    {
        $spotlights = Spotlight::with(['user'])
            ->where('moderation_status', 'approved')
            ->where('status', 'approved')
            ->where('created_at', '>=', now()->subDays(7))
            ->orderBy('likes_count', 'desc')
            ->orderBy('comments_count', 'desc')
            ->paginate($request->per_page ?? 15);

        if (auth('sanctum')->check()) {
            $spotlights->getCollection()->transform(function ($spotlight) {
                $spotlight->is_liked = SpotlightLike::where('spotlight_id', $spotlight->id)
                    ->where('user_id', auth()->id())
                    ->exists();
                return $spotlight;
            });
        }

        return response()->json([
            'success' => true,
            'data' => $spotlights
        ]);
    }
}
