<?php

namespace App\Http\Controllers\API\V1\Spotlight;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\Spotlight;
use Illuminate\Http\Request;

class SpotlightFlagController extends Controller
{
    /**
     * Flag spotlight for review (Business owners/managers)
     */
    public function flag(Request $request, $id)
    {
        $validated = $request->validate([
            'reason' => 'required|string|max:500',
        ]);

        $spotlight = Spotlight::findOrFail($id);

        // Check if user is associated with the venue/business
        $user = auth()->user();
        if (!$this->canFlagSpotlight($user, $spotlight)) {
            return response()->json([
                'success' => false,
                'message' => 'You do not have permission to flag this spotlight',
            ], 403);
        }

        // Check if already flagged
        if ($spotlight->isFlagged()) {
            return response()->json([
                'success' => false,
                'message' => 'This spotlight has already been flagged',
            ], 400);
        }

        $spotlight->flagForReview($user, $validated['reason']);

        AuditLog::logAction('spotlight.flagged', $spotlight, null, null, [
            'reason' => $validated['reason'],
            'flagged_by' => $user->id,
        ]);

        // TODO: Notify admins

        return response()->json([
            'success' => true,
            'message' => 'Spotlight flagged for review. It will be hidden until reviewed by admins.',
        ]);
    }

    /**
     * Get flagged spotlights (Admin only)
     */
    public function getFlagged(Request $request)
    {
        $query = Spotlight::with(['user', 'venue', 'flagger', 'reviewer'])
            ->whereIn('flag_status', ['flagged', 'under_review']);

        if ($request->status) {
            $query->where('flag_status', $request->status);
        }

        $spotlights = $query->orderBy('flagged_at', 'desc')->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $spotlights,
        ]);
    }

    /**
     * Review flagged spotlight (Admin)
     */
    public function review(Request $request, $id)
    {
        $validated = $request->validate([
            'action' => 'required|in:approve,remove',
            'notes' => 'nullable|string|max:1000',
        ]);

        $spotlight = Spotlight::findOrFail($id);

        if (!$spotlight->isFlagged()) {
            return response()->json([
                'success' => false,
                'message' => 'This spotlight is not flagged',
            ], 400);
        }

        if ($validated['action'] === 'approve') {
            $spotlight->clearFlag(auth()->user());

            AuditLog::logAction('spotlight.flag_cleared', $spotlight, null, null, [
                'notes' => $validated['notes'] ?? null,
            ]);

            $message = 'Spotlight flag cleared. It is now visible again.';
        } else {
            $spotlight->update([
                'status' => 'removed',
                'moderation_status' => 'rejected',
                'moderation_notes' => $validated['notes'] ?? 'Removed due to flag',
                'moderated_by' => auth()->id(),
                'moderated_at' => now(),
                'reviewed_by' => auth()->id(),
                'reviewed_at' => now(),
            ]);

            AuditLog::logAction('spotlight.removed', $spotlight, null, null, [
                'reason' => 'Flagged and removed',
                'notes' => $validated['notes'] ?? null,
            ]);

            $message = 'Spotlight removed from platform.';
        }

        return response()->json([
            'success' => true,
            'message' => $message,
        ]);
    }

    /**
     * Check if user can flag a spotlight
     */
    private function canFlagSpotlight($user, $spotlight): bool
    {
        // Admin can flag any spotlight
        if ($user->hasRole(['admin', 'super_admin'])) {
            return true;
        }

        // Business owner/manager can flag spotlights for their venues
        if ($spotlight->venue_id) {
            $business = $user->businesses()->first();
            if ($business) {
                return $business->venues()->where('id', $spotlight->venue_id)->exists();
            }
        }

        return false;
    }
}
