<?php

namespace App\Http\Controllers\API\V1\Support;

use App\Http\Controllers\Controller;
use App\Models\SupportTicket;
use App\Models\SupportTicketReply;
use Illuminate\Http\Request;

class SupportController extends Controller
{
    /**
     * Get all support tickets for current user
     */
    public function index(Request $request)
    {
        $query = SupportTicket::with(['replies' => function ($q) {
            $q->latest()->limit(1);
        }])->where('user_id', auth()->id());

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->priority) {
            $query->where('priority', $request->priority);
        }

        $tickets = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $tickets,
        ]);
    }

    /**
     * Create new support ticket
     */
    public function create(Request $request)
    {
        $validated = $request->validate([
            'department' => 'required|in:general,technical,billing,account,business',
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:5000',
            'priority' => 'nullable|in:low,normal,high,urgent',
        ]);

        $ticket = SupportTicket::create([
            'user_id' => auth()->id(),
            'department' => $validated['department'],
            'subject' => $validated['subject'],
            'priority' => $validated['priority'] ?? 'normal',
            'status' => 'open',
        ]);

        $ticket->replies()->create([
            'user_id' => auth()->id(),
            'message' => $validated['message'],
            'is_staff' => false,
        ]);

        // TODO: Send notification to support team

        return response()->json([
            'success' => true,
            'message' => 'Support ticket created successfully',
            'data' => $ticket->load('replies'),
        ], 201);
    }

    /**
     * Get single ticket with all replies
     */
    public function show($id)
    {
        $ticket = SupportTicket::with(['replies.user', 'assignedTo'])
            ->where('user_id', auth()->id())
            ->findOrFail($id);

        // Mark as read
        $ticket->update(['last_read_at' => now()]);

        return response()->json([
            'success' => true,
            'data' => $ticket,
        ]);
    }

    /**
     * Reply to ticket
     */
    public function reply(Request $request, $id)
    {
        $validated = $request->validate([
            'message' => 'required|string|max:5000',
        ]);

        $ticket = SupportTicket::where('user_id', auth()->id())
            ->findOrFail($id);

        if ($ticket->status === 'closed') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot reply to closed ticket',
            ], 400);
        }

        $reply = $ticket->replies()->create([
            'user_id' => auth()->id(),
            'message' => $validated['message'],
            'is_staff' => false,
        ]);

        $ticket->update([
            'status' => 'open',
            'updated_at' => now(),
        ]);

        // TODO: Notify assigned staff

        return response()->json([
            'success' => true,
            'message' => 'Reply sent successfully',
            'data' => $reply,
        ]);
    }

    /**
     * Close ticket
     */
    public function close($id)
    {
        $ticket = SupportTicket::where('user_id', auth()->id())
            ->findOrFail($id);

        $ticket->update([
            'status' => 'closed',
            'closed_at' => now(),
            'closed_by' => auth()->id(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Ticket closed successfully',
        ]);
    }

    /**
     * Reopen ticket
     */
    public function reopen($id)
    {
        $ticket = SupportTicket::where('user_id', auth()->id())
            ->where('status', 'closed')
            ->findOrFail($id);

        $ticket->update([
            'status' => 'open',
            'closed_at' => null,
            'closed_by' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Ticket reopened successfully',
        ]);
    }

    /**
     * Submit feedback
     */
    public function submitFeedback(Request $request)
    {
        $validated = $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'feedback' => 'required|string|max:2000',
            'category' => 'nullable|in:app,features,support,general',
        ]);

        $feedback = \App\Models\Feedback::create([
            'user_id' => auth()->id(),
            'rating' => $validated['rating'],
            'feedback' => $validated['feedback'],
            'category' => $validated['category'] ?? 'general',
        ]);

        // TODO: Notify admins if rating is low

        return response()->json([
            'success' => true,
            'message' => 'Thank you for your feedback!',
        ]);
    }

    /**
     * Submit contact form
     */
    public function submitContact(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'phone' => 'nullable|string|max:20',
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:5000',
            'type' => 'nullable|in:inquiry,support,business,partnership',
        ]);

        $contact = \App\Models\ContactMessage::create($validated);

        // TODO: Send email notification to admins
        // TODO: Send confirmation email to user

        return response()->json([
            'success' => true,
            'message' => 'Your message has been sent. We will get back to you soon!',
        ]);
    }
}
