<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AdminActivityLog;
use App\Models\User;
use Illuminate\Http\Request;

class AdminActivityController extends Controller
{
    /**
     * Display admin activity logs
     */
    public function index(Request $request)
    {
        $query = AdminActivityLog::with('adminUser');

        // Filter by action
        if ($request->filled('action')) {
            $query->where('action', $request->action);
        }

        // Filter by entity type
        if ($request->filled('entity_type')) {
            $query->where('entity_type', $request->entity_type);
        }

        // Filter by admin user
        if ($request->filled('admin_user_id')) {
            $query->where('admin_user_id', $request->admin_user_id);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Show only sensitive operations
        if ($request->boolean('sensitive_only')) {
            $query->where('is_sensitive', true);
        }

        // Show only failed operations
        if ($request->boolean('failed_only')) {
            $query->where('status', 'failed');
        }

        // Date range filter
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Search by description or entity name
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('description', 'like', "%{$search}%")
                  ->orWhere('entity_name', 'like', "%{$search}%");
            });
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $logs = $query->paginate(50);

        // Statistics
        $stats = [
            'total_logs' => AdminActivityLog::count(),
            'today' => AdminActivityLog::whereDate('created_at', today())->count(),
            'this_week' => AdminActivityLog::where('created_at', '>=', now()->subWeek())->count(),
            'sensitive' => AdminActivityLog::where('is_sensitive', true)->count(),
            'failed' => AdminActivityLog::where('status', 'failed')->count(),
        ];

        $actions = AdminActivityLog::distinct('action')->pluck('action')->toArray();
        $entityTypes = AdminActivityLog::distinct('entity_type')->pluck('entity_type')->toArray();
        $admins = User::whereHas('roles', function ($q) {
            $q->whereIn('name', ['admin', 'super_admin']);
        })->select('id', 'first_name', 'last_name', 'email')->get();

        return view('admin.activities.index', compact('logs', 'stats', 'actions', 'entityTypes', 'admins'));
    }

    /**
     * Show detailed activity log
     */
    public function show($id)
    {
        $log = AdminActivityLog::with('adminUser')->findOrFail($id);
//        return json response for API or view for web
        return response()->json([
            'success' => true,
            'data' => $log,
        ]);
    }

    /**
     * Export activity logs
     */
    public function export(Request $request)
    {
        $query = AdminActivityLog::with('adminUser');

        // Apply same filters as index
        if ($request->filled('action')) {
            $query->where('action', $request->action);
        }

        if ($request->filled('entity_type')) {
            $query->where('entity_type', $request->entity_type);
        }

        if ($request->filled('admin_user_id')) {
            $query->where('admin_user_id', $request->admin_user_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $logs = $query->orderBy('created_at', 'desc')->get();

        // Generate CSV
        $csvContent = $this->generateCsv($logs);

        return response()->streamDownload(
            fn () => print $csvContent,
            'admin-activity-' . now()->format('Y-m-d-H-i-s') . '.csv',
            [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => 'attachment; filename="admin-activity-export.csv"',
            ]
        );
    }

    /**
     * Generate CSV content from logs
     */
    private function generateCsv($logs)
    {
        $output = fopen('php://memory', 'r+');

        // Write header
        fputcsv($output, [
            'Timestamp',
            'Admin User',
            'Action',
            'Entity Type',
            'Entity Name',
            'Status',
            'Sensitive',
            'IP Address',
            'Description',
        ]);

        // Write data
        foreach ($logs as $log) {
            fputcsv($output, [
                $log->created_at->format('Y-m-d H:i:s'),
                $log->adminUser->name ?? 'N/A',
                $log->action,
                $log->entity_type,
                $log->entity_name,
                $log->status,
                $log->is_sensitive ? 'Yes' : 'No',
                $log->ip_address,
                $log->description,
            ]);
        }

        rewind($output);
        return stream_get_contents($output);
    }

    /**
     * Get activity statistics
     */
    public function statistics(Request $request)
    {
        $days = $request->get('days', 30);

        // Activity by day
        $activityByDay = AdminActivityLog::selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->where('created_at', '>=', now()->subDays($days))
            ->groupBy('date')
            ->orderBy('date', 'asc')
            ->get();

        // Activity by action
        $activityByAction = AdminActivityLog::selectRaw('action, COUNT(*) as count')
            ->where('created_at', '>=', now()->subDays($days))
            ->groupBy('action')
            ->orderBy('count', 'desc')
            ->get();

        // Activity by entity type
        $activityByEntity = AdminActivityLog::selectRaw('entity_type, COUNT(*) as count')
            ->where('created_at', '>=', now()->subDays($days))
            ->groupBy('entity_type')
            ->orderBy('count', 'desc')
            ->get();

        // Top admins by activity
        $topAdmins = AdminActivityLog::selectRaw('admin_user_id, COUNT(*) as count')
            ->where('created_at', '>=', now()->subDays($days))
            ->groupBy('admin_user_id')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->with('adminUser')
            ->get();

        // Failed operations
        $failedCount = AdminActivityLog::where('status', 'failed')
            ->where('created_at', '>=', now()->subDays($days))
            ->count();

        // Sensitive operations
        $sensitiveCount = AdminActivityLog::where('is_sensitive', true)
            ->where('created_at', '>=', now()->subDays($days))
            ->count();

        return response()->json([
            'success' => true,
            'data' => [
                'activity_by_day' => $activityByDay,
                'activity_by_action' => $activityByAction,
                'activity_by_entity' => $activityByEntity,
                'top_admins' => $topAdmins,
                'failed_count' => $failedCount,
                'sensitive_count' => $sensitiveCount,
            ],
        ]);
    }

    /**
     * Get activity audit trail for specific entity
     */
    public function entityAuditTrail($entityType, $entityId)
    {
        $logs = AdminActivityLog::where('entity_type', $entityType)
            ->where('entity_id', $entityId)
            ->with('adminUser')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $logs,
        ]);
    }

    /**
     * Get activity timeline for user
     */
    public function userActivityTimeline($userId)
    {
        $logs = AdminActivityLog::where('admin_user_id', $userId)
            ->orderBy('created_at', 'desc')
            ->limit(100)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $logs,
        ]);
    }

    /**
     * Purge old activity logs
     */
    public function purgeOldLogs(Request $request)
    {
        $request->validate([
            'days' => 'required|integer|min:1|max:365',
        ]);

        $deletedCount = AdminActivityLog::where('created_at', '<', now()->subDays($request->days))
            ->delete();

        return response()->json([
            'success' => true,
            'message' => "{$deletedCount} old activity logs deleted",
            'deleted_count' => $deletedCount,
        ]);
    }
}

