<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SystemConfiguration;
use App\Models\ConfigurationVersion;
use App\Models\AdminActivityLog;
use App\Helpers\ConfigHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AdminConfigurationController extends Controller
{
    const CACHE_TTL = 3600; // 1 hour
    const CONFIG_CACHE_PREFIX = 'config.';
    const GROUP_CACHE_PREFIX = 'config.group.';

    /**
     * Display system configurations with versioning
     */
    public function index()
    {
        try {
            // Load configurations grouped by category
            $configurations = SystemConfiguration::orderBy('group')
                ->orderBy('sort_order')
                ->get()
                ->groupBy('group');

            // Configuration group labels
            $groups = $this->getConfigurationGroups();

            // Load version history
            $versions = ConfigurationVersion::with('creator')
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get();

            $currentVersion = ConfigurationVersion::where('is_current', true)->first();

            return view('admin.configurations.improved', compact(
                'configurations',
                'groups',
                'versions',
                'currentVersion'
            ));
        } catch (\Exception $e) {
            Log::error('Failed to load configuration page', [
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to load configurations: ' . $e->getMessage());
        }
    }

    /**
     * Update single configuration with audit trail
     */
    public function update(Request $request)
    {
        try {
            $validated = $request->validate([
                'key' => 'required|string',
                'value' => 'nullable|string',
            ]);

            // Find configuration by key
            $config = SystemConfiguration::where('key', $validated['key'])->first();

            if (!$config) {
                // Create new configuration if it doesn't exist
                $config = SystemConfiguration::create([
                    'key' => $validated['key'],
                    'value' => '',
                    'type' => 'string',
                    'group' => 'general',
                    'sort_order' => 0,
                ]);
            }

            $oldValue = $config->value;
            $newValue = $validated['value'];

            // Handle different data types
            $newValue = $this->castConfigValue($config->type, $newValue);

            // Update regardless of change detection (allow same value updates)
            $config->update([
                'value' => $newValue,
                'changed_by' => auth()->id(),
                'version' => ($config->version ?? 0) + 1,
            ]);

            // Log activity
            $this->logConfigChange($config, $oldValue, $newValue, 'update');

            // Clear caches
            $this->clearConfigCaches($config->key, $config->group);

            return response()->json([
                'success' => true,
                'message' => 'Configuration updated successfully',
                'data' => $config,
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            Log::error('Configuration update failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => auth()->id(),
                'key' => $validated['key'] ?? null,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update configuration: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Bulk update configurations from form
     */
    public function bulkUpdate(Request $request)
    {
        try {
            // Collect all form data
            $updates = [];

            // Get all input data
            $allInputs = $request->all();

            // Process all input fields
            foreach ($allInputs as $key => $value) {
                // Skip CSRF token and method
                if ($key === '_token' || $key === '_method') {
                    continue;
                }

                // Handle nested array inputs like payment[stripe][enabled]
                if (is_array($value)) {
                    // Flatten nested arrays: payment => [stripe => [enabled => 1]]
                    // becomes: payment.stripe.enabled => 1
                    $this->flattenArray($value, $key, $updates);
                } else {
                    // Handle direct string values
                    $updates[$key] = $value;
                }
            }

            // Debug logging
            Log::info('Bulk update received data', [
                'total_fields' => count($updates),
                'keys' => array_keys($updates),
                'all_data' => $updates,
            ]);

            if (empty($updates)) {
                return response()->json([
                    'success' => true,
                    'message' => 'No configuration fields provided',
                    'data' => [],
                ]);
            }

            DB::beginTransaction();

            $updateCount = 0;
            $failedConfigs = [];
            $changes = [];

            // Update each configuration
            foreach ($updates as $configKey => $value) {
                try {
                    // Ensure value is not an array
                    if (is_array($value)) {
                        $value = json_encode($value);
                    } else {
                        $value = (string) $value;
                    }

                    // Find or create configuration
                    $config = SystemConfiguration::firstOrCreate(
                        ['key' => $configKey],
                        [
                            'value' => '',
                            'type' => 'string',
                            'group' => 'general',
                            'sort_order' => 0,
                        ]
                    );

                    $oldValue = $config->value;

                    // Cast value based on type
                    $newValue = $this->castConfigValue($config->type, $value);

                    // Update configuration (always update)
                    $config->update([
                        'value' => $newValue,
                        'changed_by' => auth()->id(),
                        'version' => ($config->version ?? 0) + 1,
                    ]);

                    $updateCount++;

                    // Track changes only if value actually changed
                    if ($oldValue !== $newValue) {
                        $changes[$configKey] = [
                            'from' => substr($oldValue, 0, 50),
                            'to' => substr($newValue, 0, 50),
                        ];
                    }

                    // Clear cache for this config
                    $this->clearConfigCaches($configKey, $config->group);

                    Log::info('Configuration updated', [
                        'key' => $configKey,
                        'type' => $config->type,
                        'new_value' => substr($newValue, 0, 50),
                    ]);
                } catch (\Exception $e) {
                    $failedConfigs[] = [
                        'key' => $configKey,
                        'error' => $e->getMessage(),
                    ];

                    Log::error('Failed to update individual config', [
                        'key' => $configKey,
                        'value' => $value ?? null,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString(),
                    ]);
                }
            }

            // Log bulk operation
            if ($updateCount > 0) {
                $this->logBulkConfigUpdate($updateCount, $changes, $failedConfigs);
            }

            // Clear general caches
            Cache::forget('config.public');
            Cache::forget('config.all');

            DB::commit();

            $message = $updateCount > 0
                ? "{$updateCount} configuration(s) updated successfully"
                : 'No configurations were updated';

            if (!empty($failedConfigs)) {
                $message .= ' (' . count($failedConfigs) . ' failed)';
            }

            return response()->json([
                'success' => empty($failedConfigs),
                'message' => $message,
                'updated_count' => $updateCount,
                'failed_count' => count($failedConfigs),
                'failed_configs' => $failedConfigs,
                'data' => $changes,
            ], empty($failedConfigs) ? 200 : 207);
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Bulk configuration update failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => auth()->id(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Bulk update failed: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Create configuration snapshot/version
     */
    public function createSnapshot(Request $request)
    {
        try {
            $validated = $request->validate([
                'version_name' => 'required|string|max:255|unique:configuration_versions,version_name',
                'description' => 'nullable|string|max:1000',
            ]);

            // Capture current configurations
            $configurations = SystemConfiguration::pluck('value', 'key')->toArray();

            // Create version
            $version = ConfigurationVersion::create([
                'created_by' => auth()->id(),
                'configurations' => $configurations,
                'version_name' => $validated['version_name'],
                'description' => $validated['description'] ?? null,
                'is_backup' => false,
                'is_current' => false,
            ]);

            // Log activity
            AdminActivityLog::create([
                'admin_user_id' => auth()->id(),
                'action' => 'create',
                'entity_type' => 'ConfigurationVersion',
                'entity_id' => $version->id,
                'entity_name' => $validated['version_name'],
                'description' => "Created configuration snapshot: {$validated['version_name']}",
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
                'status' => 'success',
                'is_sensitive' => false,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Configuration snapshot created successfully',
                'data' => $version,
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            Log::error('Failed to create configuration snapshot', [
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to create snapshot: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get configuration versions list
     */
    public function getVersions()
    {
        try {
            $versions = ConfigurationVersion::with('creator')
                ->orderBy('created_at', 'desc')
                ->paginate(20);

            return response()->json([
                'success' => true,
                'data' => $versions,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to retrieve versions', [
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve versions',
            ], 500);
        }
    }

    /**
     * View specific version details
     */
    public function showVersion($versionId)
    {
        try {
            $version = ConfigurationVersion::with('creator')->findOrFail($versionId);
            $currentVersion = ConfigurationVersion::where('is_current', true)->first();

            $diffs = [];
            if ($currentVersion && $currentVersion->id !== $version->id) {
                $diffs = $this->calculateDifferences(
                    $version->configurations,
                    $currentVersion->configurations
                );
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'version' => $version,
                    'diffs' => $diffs,
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to show version', [
                'error' => $e->getMessage(),
                'version_id' => $versionId,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Version not found',
            ], 404);
        }
    }

    /**
     * Restore configuration from version
     */
    public function restoreVersion(Request $request, $versionId)
    {
        try {
            $version = ConfigurationVersion::findOrFail($versionId);

            $request->validate([
                'confirm' => 'required|boolean|accepted',
            ]);

            if ($version->is_current) {
                return response()->json([
                    'success' => false,
                    'message' => 'This is already the current version',
                ], 400);
            }

            DB::beginTransaction();

            // Store old values
            $oldConfigurations = SystemConfiguration::pluck('value', 'key')->toArray();

            // Restore configurations
            $restoredCount = 0;
            foreach ($version->configurations as $key => $value) {
                try {
                    $config = SystemConfiguration::where('key', $key)->first();

                    if ($config) {
                        $config->update([
                            'value' => $value,
                            'changed_by' => auth()->id(),
                            'version' => ($config->version ?? 0) + 1,
                        ]);

                        $restoredCount++;
                        $this->clearConfigCaches($key, $config->group);
                    }
                } catch (\Exception $e) {
                    Log::warning("Failed to restore config: {$key}", [
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            // Mark as current
            ConfigurationVersion::where('is_current', true)
                ->update(['is_current' => false]);

            $version->update(['is_current' => true]);

            // Log restoration
            AdminActivityLog::create([
                'admin_user_id' => auth()->id(),
                'action' => 'restore',
                'entity_type' => 'ConfigurationVersion',
                'entity_id' => $version->id,
                'entity_name' => $version->version_name,
                'description' => "Restored configuration from: {$version->version_name} ({$restoredCount} items)",
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
                'status' => 'success',
                'is_sensitive' => true,
            ]);

            // Clear all caches
            Cache::flush();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Configuration restored successfully ({$restoredCount} items restored)",
                'restored_count' => $restoredCount,
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Failed to restore configuration', [
                'error' => $e->getMessage(),
                'version_id' => $versionId,
                'user_id' => auth()->id(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to restore configuration: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Compare two configuration versions
     */
    public function compareVersions(Request $request)
    {
        try {
            $validated = $request->validate([
                'version_id_1' => 'required|exists:configuration_versions,id',
                'version_id_2' => 'required|exists:configuration_versions,id',
            ]);

            $version1 = ConfigurationVersion::findOrFail($validated['version_id_1']);
            $version2 = ConfigurationVersion::findOrFail($validated['version_id_2']);

            $diffs = $this->calculateDifferences(
                $version1->configurations,
                $version2->configurations
            );

            return response()->json([
                'success' => true,
                'data' => [
                    'version_1' => $version1,
                    'version_2' => $version2,
                    'differences' => $diffs,
                    'change_count' => count($diffs),
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to compare versions', [
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to compare versions',
            ], 500);
        }
    }

    /**
     * Delete configuration version
     */
    public function deleteVersion($versionId)
    {
        try {
            $version = ConfigurationVersion::findOrFail($versionId);

            if ($version->is_current) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete the current configuration version',
                ], 400);
            }

            $versionName = $version->version_name;
            $version->delete();

            // Log activity
            AdminActivityLog::create([
                'admin_user_id' => auth()->id(),
                'action' => 'delete',
                'entity_type' => 'ConfigurationVersion',
                'entity_id' => $versionId,
                'entity_name' => $versionName,
                'description' => "Deleted configuration version: {$versionName}",
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
                'status' => 'success',
                'is_sensitive' => false,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Configuration version deleted successfully',
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to delete version', [
                'error' => $e->getMessage(),
                'version_id' => $versionId,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete version: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * HELPER METHODS
     */

    /**
     * Flatten nested array to dot notation
     */
    private function flattenArray($array, $prefix, &$result)
    {
        foreach ($array as $key => $value) {
            $newKey = $prefix . '.' . $key;

            if (is_array($value)) {
                // Recursively flatten nested arrays
                $this->flattenArray($value, $newKey, $result);
            } else {
                // Handle checkbox values
                if ($value === 'on' || is_null($value)) {
                    $value = '1';
                } elseif ($value === false) {
                    $value = '0';
                }

                $result[$newKey] = $value;
            }
        }
    }

    /**
     * Cast configuration value to correct type
     */
    private function castConfigValue($type, $value)
    {
        if (is_null($value) || $value === '') {
            return null;
        }

        switch ($type) {
            case 'boolean':
                return ($value === '1' || $value === 1 || $value === true || $value === 'true') ? '1' : '0';

            case 'integer':
                return (int) $value;

            case 'json':
            case 'array':
                return is_array($value) ? json_encode($value) : $value;

            default:
                return (string) $value;
        }
    }

    /**
     * Clear configuration caches
     */
    private function clearConfigCaches($key, $group = null)
    {
        Cache::forget(self::CONFIG_CACHE_PREFIX . $key);

        if ($group) {
            Cache::forget(self::GROUP_CACHE_PREFIX . $group);
        }

        Cache::forget('config.public');
        Cache::forget('config.all');
    }

    /**
     * Log configuration change
     */
    private function logConfigChange($config, $oldValue, $newValue, $action)
    {
        AdminActivityLog::create([
            'admin_user_id' => auth()->id(),
            'action' => $action,
            'entity_type' => 'SystemConfiguration',
            'entity_id' => $config->id,
            'entity_name' => $config->key,
            'old_values' => ['value' => $oldValue],
            'new_values' => ['value' => $newValue],
            'description' => "Updated {$config->key}",
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'status' => 'success',
            'is_sensitive' => !($config->is_public ?? true),
        ]);
    }

    /**
     * Log bulk configuration update
     */
    private function logBulkConfigUpdate($updateCount, $changes, $failedConfigs)
    {
        AdminActivityLog::create([
            'admin_user_id' => auth()->id(),
            'action' => 'bulk_update',
            'entity_type' => 'SystemConfiguration',
            'entity_id' => null,
            'entity_name' => 'Multiple Configurations',
            'changes' => $changes,
            'description' => "Bulk updated {$updateCount} configurations" .
                (count($failedConfigs) > 0 ? " ({$failedConfigs} failed)" : ''),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'status' => empty($failedConfigs) ? 'success' : 'partial_success',
            'is_sensitive' => false,
        ]);
    }

    /**
     * Calculate differences between two configuration sets
     */
    private function calculateDifferences($config1, $config2)
    {
        $diffs = [];

        // Find changed and added items
        foreach ($config1 as $key => $value) {
            if (!isset($config2[$key])) {
                $diffs[$key] = [
                    'status' => 'added',
                    'from' => null,
                    'to' => $value,
                ];
            } elseif ($config2[$key] !== $value) {
                $diffs[$key] = [
                    'status' => 'changed',
                    'from' => $config2[$key],
                    'to' => $value,
                ];
            }
        }

        // Find removed items
        foreach ($config2 as $key => $value) {
            if (!isset($config1[$key])) {
                $diffs[$key] = [
                    'status' => 'removed',
                    'from' => $value,
                    'to' => null,
                ];
            }
        }

        return $diffs;
    }

    /**
     * Get configuration groups
     */
    private function getConfigurationGroups()
    {
        return [
            'general' => 'General Settings',
            'authentication' => 'Authentication & Security',
            'referral' => 'Referral System',
            'spotlight' => 'Spotlight Settings',
            'venue' => 'Venue Management',
            'offer' => 'Offer Management',
            'notification' => 'Notifications',
            'sms' => 'SMS Provider',
            'email' => 'Email Configuration',
            'chat' => 'Chat & Messaging',
            'gdpr' => 'GDPR Compliance',
            'payment' => 'Payment Settings',
            'stripe' => 'Stripe Payment',
            'paypal' => 'PayPal Payment',
            'security' => 'Security Settings',
            'google' => 'Google OAuth',
            'facebook' => 'Facebook OAuth',
            'push' => 'Push Notifications',
        ];
    }
}


