<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Notifications\Notification;
use Illuminate\Http\Request;

class AdminNotificationController extends Controller
{
    /**
     * Display all admin notifications
     */
    public function index()
    {
        $admin = auth()->user();

        // Ensure user is authenticated
        if (!$admin) {
            return redirect()->route('admin.login');
        }

        // Initialize unreadCount with default value
        $unreadCount = 0;
        $notifications = collect([]);

        // Get all notifications for admin
        try {
            $notifications = $admin->notifications()
                ->orderBy('read_at')
                ->orderBy('created_at', 'desc')
                ->paginate(20);

            // Get unread count safely with multiple fallbacks
            if (method_exists($admin, 'unreadNotifications') && $admin->unreadNotifications) {
                $unreadCount = $admin->unreadNotifications->count();
            } else {
                // Fallback: count where read_at is null
                $unreadCount = $admin->notifications()
                    ->whereNull('read_at')
                    ->count();
            }
        } catch (\Exception $e) {
            // Log error but don't break the page
            \Log::error('Error loading admin notifications: ' . $e->getMessage());
            $unreadCount = 0;
            $notifications = collect([]);
        }

        return view('admin.notifications.index', [
            'notifications' => $notifications,
            'unreadCount' => $unreadCount,
        ]);
    }

    /**
     * Mark notification as read
     */
    public function markAsRead($id)
    {
        $admin = auth()->user();
        $notification = $admin->notifications()->findOrFail($id);
        $notification->markAsRead();

        return response()->json(['success' => true]);
    }

    /**
     * Mark all notifications as read
     */
    public function markAllAsRead()
    {
        auth()->user()->unreadNotifications->markAsRead();

        return response()->json(['success' => true]);
    }

    /**
     * Delete notification
     */
    public function destroy($id)
    {
        $admin = auth()->user();
        $notification = $admin->notifications()->findOrFail($id);
        $notification->delete();

        return response()->json(['success' => true]);
    }

    /**
     * Get notifications count (for header)
     */
    public function getCount()
    {
        $count = auth()->user()->unreadNotifications->count();

        return response()->json([
            'unread_count' => $count,
        ]);
    }
}

