<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\User;
use App\Models\Offer;
use App\Models\Redemption;
use App\Models\Review;
use App\Services\AnalyticsService;
use Illuminate\Http\Request;
use Carbon\Carbon;

class AnalyticsController extends Controller
{
    /**
     * Display advanced analytics dashboard
     */
    public function index(Request $request)
    {
        // Check permission
        if (!auth()->user()->hasAnyRole(['super_admin', 'admin'])) {
            abort(403, 'Unauthorized');
        }

        $period = $request->get('period', '30');
        $startDate = Carbon::now()->subDays($period);

        // Overall Statistics
        $stats = [
            'total_users' => User::count(),
            'active_users' => User::where('status', 'active')->count(),
            'total_businesses' => Business::count(),
            'active_businesses' => Business::where('status', 'active')->count(),
            'total_offers' => Offer::count(),
            'active_offers' => Offer::where('status', 'active')->count(),
            'total_redemptions' => Redemption::count(),
            'redemptions_period' => Redemption::where('created_at', '>=', $startDate)->count(),
            'avg_rating' => Review::avg('rating') ?? 0,
            'total_reviews' => Review::count(),
        ];

        // Redemption Analytics
        $redemptions = [
            'total' => Redemption::count(),
            'this_period' => Redemption::where('created_at', '>=', $startDate)->count(),
            'by_status' => Redemption::selectRaw('status, COUNT(*) as count')->groupBy('status')->get(),
            'trending' => $this->getRedemptionTrends($period),
        ];

        // Business Performance
        $businessPerformance = Business::with('staff', 'venues', 'offers')
            ->withCount(['redemptions' => function ($query) use ($startDate) {
                $query->where('redemptions.created_at', '>=', $startDate);
            }])
            ->withAvg('reviews', 'rating')
            ->orderBy('redemptions_count', 'desc')
            ->limit(10)
            ->get();

        // Top Offers
        $topOffers = Offer::withCount(['redemptions' => function ($q) use ($startDate) {
            $q->where('redemptions.created_at', '>=', $startDate);
        }])
            ->orderBy('redemptions_count', 'desc')
            ->limit(10)
            ->get();

        // Revenue Metrics
        $revenue = $this->getRevenueMetrics($startDate);

        // User Growth
        $userGrowth = $this->getUserGrowthTrends($period);

        // Business Growth
        $businessGrowth = $this->getBusinessGrowthTrends($period);

        $analytics = AnalyticsService::getDashboardAnalytics();

        return view('admin.analytics.dashboard', array_merge($analytics, [
            'stats' => $stats,
            'redemptions' => $redemptions,
            'businessPerformance' => $businessPerformance,
            'topOffers' => $topOffers,
            'revenue' => $revenue,
            'userGrowth' => $userGrowth,
            'businessGrowth' => $businessGrowth,
            'period' => $period,
        ]));
    }

    /**
     * Get redemptions report
     */
    public function redemptionsReport(Request $request)
    {
        if (!auth()->user()->hasAnyRole(['super_admin', 'admin'])) {
            abort(403, 'Unauthorized');
        }

        $period = $request->get('period', '30');
        $startDate = Carbon::now()->subDays($period);
        $status = $request->get('status', null);

        $query = Redemption::with('user', 'offer', 'offer.business')
            ->where('created_at', '>=', $startDate);

        if ($status) {
            $query->where('status', $status);
        }

        $redemptions = $query->orderBy('created_at', 'desc')
            ->paginate(50);

        return view('admin.analytics.redemptions-report', [
            'redemptions' => $redemptions,
            'period' => $period,
            'status' => $status,
        ]);
    }

    /**
     * Get business performance report
     */
    public function businessReport(Request $request)
    {
        if (!auth()->user()->hasAnyRole(['super_admin', 'admin'])) {
            abort(403, 'Unauthorized');
        }

        $period = $request->get('period', '30');
        $startDate = Carbon::now()->subDays($period);
        $sortBy = $request->get('sort', 'redemptions');

        $businesses = Business::with('owner', 'staff', 'venues', 'offers')
            ->withCount(['staff', 'venues', 'offers'])
            ->withCount(['redemptions' => function ($q) use ($startDate) {
                $q->where('redemptions.created_at', '>=', $startDate);
            }])
            ->withAvg('reviews', 'rating');

        // Sort by requested column
        if ($sortBy === 'redemptions') {
            $businesses = $businesses->orderBy('redemptions_count', 'desc');
        } elseif ($sortBy === 'rating') {
            $businesses = $businesses->orderByDesc('reviews_avg_rating');
        } elseif ($sortBy === 'revenue') {
            $businesses = $businesses->orderBy('redemptions_count', 'desc');
        }

        $businesses = $businesses->paginate(50);

        return view('admin.analytics.business-report', [
            'businesses' => $businesses,
            'period' => $period,
            'sortBy' => $sortBy,
        ]);
    }

    /**
     * Get user growth report
     */
    public function userReport(Request $request)
    {
        if (!auth()->user()->hasAnyRole(['super_admin', 'admin'])) {
            abort(403, 'Unauthorized');
        }

        $period = $request->get('period', '30');
        $startDate = Carbon::now()->subDays($period);
        $accountType = $request->get('account_type', null);

        $query = User::where('created_at', '>=', $startDate);

        if ($accountType) {
            $query->where('account_type', $accountType);
        }

        $users = $query->withCount('redemptions')
            ->orderBy('created_at', 'desc')
            ->paginate(50);

        return view('admin.analytics.user-report', [
            'users' => $users,
            'period' => $period,
            'accountType' => $accountType,
        ]);
    }

    /**
     * Get redemption trends
     */
    private function getRedemptionTrends($period)
    {
        $trends = [];
        for ($i = $period - 1; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            $count = Redemption::whereDate('created_at', $date->toDateString())->count();
            $trends[] = [
                'date' => $date->format('M d'),
                'count' => $count,
            ];
        }
        return $trends;
    }

    /**
     * Get revenue metrics
     */
    private function getRevenueMetrics($startDate)
    {
        $redemptions = Redemption::where('created_at', '>=', $startDate)
            ->with('offer')
            ->get();

        $total = $redemptions->sum(function ($r) {
            return $r->offer->discount_value ?? 0;
        });

        return [
            'total' => $total,
            'count' => $redemptions->count(),
            'average' => $redemptions->count() > 0 ? $total / $redemptions->count() : 0,
        ];
    }

    /**
     * Get user growth trends
     */
    private function getUserGrowthTrends($period)
    {
        $trends = [];
        for ($i = $period - 1; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            $count = User::whereDate('created_at', '<=', $date->toDateString())->count();
            $trends[] = [
                'date' => $date->format('M d'),
                'count' => $count,
            ];
        }
        return $trends;
    }

    /**
     * Get business growth trends
     */
    private function getBusinessGrowthTrends($period)
    {
        $trends = [];
        for ($i = $period - 1; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            $count = Business::whereDate('created_at', '<=', $date->toDateString())->count();
            $trends[] = [
                'date' => $date->format('M d'),
                'count' => $count,
            ];
        }
        return $trends;
    }

    /**
     * Export analytics data
     */
    public function export(Request $request)
    {
        if (!auth()->user()->hasAnyRole(['super_admin', 'admin'])) {
            abort(403, 'Unauthorized');
        }

        $format = $request->get('format', 'csv');
        $type = $request->get('type', 'redemptions');
        $period = $request->get('period', '30');

        if ($format === 'csv') {
            return $this->exportToCSV($type, $period);
        }

        return back()->with('error', 'Invalid export format');
    }

    /**
     * Export to CSV
     */
    private function exportToCSV($type, $period)
    {
        $startDate = Carbon::now()->subDays($period);

        if ($type === 'redemptions') {
            $data = Redemption::with('user', 'offer', 'offer.business')
                ->where('created_at', '>=', $startDate)
                ->get();

            $filename = 'redemptions_' . date('Y-m-d_H-i-s') . '.csv';
            $headers = ['User', 'Offer', 'Business', 'Status', 'Date'];

            return $this->generateCSV($filename, $headers, $data, 'redemptions');
        }

        return back()->with('error', 'Invalid export type');
    }

    /**
     * Generate CSV file
     */
    private function generateCSV($filename, $headers, $data, $type)
    {
        $callback = function() use ($headers, $data, $type) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $headers);

            foreach ($data as $row) {
                if ($type === 'redemptions') {
                    fputcsv($file, [
                        $row->user->first_name . ' ' . $row->user->last_name,
                        $row->offer->title,
                        $row->offer->business->business_name,
                        ucfirst($row->status),
                        $row->created_at->format('Y-m-d H:i:s'),
                    ]);
                }
            }

            fclose($file);
        };

        return response()->stream($callback, 200, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    /**
     * Get analytics data as JSON
     */
    public function getData()
    {
        if (!auth()->user()->hasAnyRole(['super_admin', 'admin'])) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $analytics = AnalyticsService::getDashboardAnalytics();

        return response()->json([
            'success' => true,
            'data' => $analytics,
        ]);
    }

    /**
     * Get activity trends for chart
     */
    public function activityTrends(Request $request)
    {
        if (!auth()->user()->hasPermissionTo('activities.view')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $days = $request->get('days', 30);
        $trends = AnalyticsService::getActivityTrends();

        return response()->json([
            'success' => true,
            'data' => array_slice($trends, -$days),
        ]);
    }

    /**
     * Get user analytics
     */
    public function userAnalytics()
    {
        if (!auth()->user()->hasPermissionTo('users.view')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $analytics = AnalyticsService::getUserAnalytics();

        return response()->json([
            'success' => true,
            'data' => $analytics,
        ]);
    }

    /**
     * Get admin performance metrics
     */
    public function adminPerformance()
    {
        if (!auth()->user()->hasPermissionTo('activities.view')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $performance = AnalyticsService::getAdminPerformance();

        return response()->json([
            'success' => true,
            'data' => $performance,
        ]);
    }

    /**
     * Get security metrics
     */
    public function securityMetrics()
    {
        if (!auth()->user()->hasPermissionTo('activities.view')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $security = AnalyticsService::getSecurityMetrics();

        return response()->json([
            'success' => true,
            'data' => $security,
        ]);
    }

    /**
     * Get system health status
     */
    public function systemHealth()
    {
        if (!auth()->user()->hasAnyRole(['super_admin', 'admin'])) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $health = AnalyticsService::getSystemHealth();

        return response()->json([
            'success' => true,
            'data' => $health,
        ]);
    }

    /**
     * Get activity breakdown
     */
    public function activityBreakdown(Request $request)
    {
        if (!auth()->user()->hasPermissionTo('activities.view')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $days = $request->get('days', 7);
        $breakdown = AnalyticsService::getActivityBreakdown($days);

        return response()->json([
            'success' => true,
            'data' => $breakdown,
        ]);
    }

    /**
     * Get hourly activity pattern
     */
    public function hourlyPattern()
    {
        if (!auth()->user()->hasPermissionTo('activities.view')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $pattern = AnalyticsService::getHourlyActivityPattern();

        return response()->json([
            'success' => true,
            'data' => $pattern,
        ]);
    }

    /**
     * Get user retention metrics
     */
    public function userRetention()
    {
        if (!auth()->user()->hasPermissionTo('users.view')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $retention = AnalyticsService::getUserRetention();

        return response()->json([
            'success' => true,
            'data' => $retention,
        ]);
    }
}

