<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AdminSession;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\User;

class AuthController extends Controller
{
    /**
     * Show admin login form
     */
    public function showLoginForm()
    {
        return view('admin.auth.login');
    }

    /**
     * Handle admin login
     */
    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        $user = User::where('email', $credentials['email'])->first();

        if (!$user || !Hash::check($credentials['password'], $user->password)) {
            return back()->withErrors([
                'email' => 'The provided credentials do not match our records.',
            ])->withInput();
        }

        // Check if user has admin role (super_admin or admin)
        if (!$user->isAdminUser()) {
            return back()->withErrors([
                'email' => 'You do not have admin access.',
            ])->withInput();
        }

        // Check if account is active
        if ($user->status !== 'active') {
            return back()->withErrors([
                'email' => 'Your account is not active.',
            ])->withInput();
        }

        // Log the user in
        Auth::login($user, $request->boolean('remember'));

        // Update last login
        $user->update(['last_login_at' => now(), 'last_active_at' => now()]);

        $request->session()->regenerate();

        // Create admin session record
        $sessionToken = bin2hex(random_bytes(32));
        $session = AdminSession::create([
            'user_id' => $user->id,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'session_token' => $sessionToken,
            'login_at' => now(),
            'last_activity_at' => now(),
            'is_active' => true,
            'login_method' => 'email',
            'metadata' => [
                'browser' => $this->getBrowserInfo($request->userAgent()),
                'device' => $this->getDeviceInfo($request->userAgent()),
            ],
        ]);

        // Store session ID in session
        session(['admin_session_id' => $session->id]);

        return redirect()->intended(route('admin.dashboard'));
    }

    /**
     * Handle admin logout
     */
    public function logout(Request $request)
    {
        // Mark admin session as inactive
        $sessionId = session('admin_session_id');
        if ($sessionId) {
            AdminSession::find($sessionId)?->update([
                'logout_at' => now(),
                'is_active' => false,
            ]);
        }

        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('admin.login')
            ->with('success', 'You have been logged out successfully.');
    }

    /**
     * Extract browser info from user agent
     */
    private function getBrowserInfo($userAgent)
    {
        if (strpos($userAgent, 'Chrome') !== false) {
            return 'Chrome';
        } elseif (strpos($userAgent, 'Firefox') !== false) {
            return 'Firefox';
        } elseif (strpos($userAgent, 'Safari') !== false) {
            return 'Safari';
        } elseif (strpos($userAgent, 'Edge') !== false) {
            return 'Edge';
        } else {
            return 'Unknown';
        }
    }

    /**
     * Extract device info from user agent
     */
    private function getDeviceInfo($userAgent)
    {
        if (strpos($userAgent, 'Mobile') !== false) {
            return 'Mobile';
        } elseif (strpos($userAgent, 'Tablet') !== false) {
            return 'Tablet';
        } else {
            return 'Desktop';
        }
    }
}
