<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Chat;
use App\Models\Message;
use Illuminate\Http\Request;

class ChatController extends Controller
{
    /**
     * Display a listing of chats
     */
    public function index()
    {
        $chats = Chat::with('participants')
            ->when(request('search'), function ($query) {
                $query->where('name', 'like', '%' . request('search') . '%');
            })
            ->when(request('type'), function ($query) {
                $query->where('type', request('type'));
            })
            ->when(request('status'), function ($query) {
                if (request('status') === 'active') {
                    $query->where('is_active', true);
                } elseif (request('status') === 'inactive') {
                    $query->where('is_active', false);
                }
            })
            ->withCount('participants')
            ->withCount('messages')
            ->paginate(50);

        $stats = [
            'total' => Chat::count(),
            'active' => Chat::where('is_active', true)->count(),
            'messages' => Message::count(),
            'active_users' => Chat::join('chat_participants', 'chats.id', '=', 'chat_participants.chat_id')
                ->where('chats.is_active', true)
                ->distinct()
                ->count('chat_participants.user_id'),
        ];

        return view('admin.chats.index', compact('chats', 'stats'));
    }

    /**
     * Display the specified chat
     */
    public function show(Chat $chat)
    {
        $chat->load('participants', 'messages.sender');
        $messages = $chat->messages()->with('sender')->latest()->paginate(50);

        return view('admin.chats.show', compact('chat', 'messages'));
    }

    /**
     * Disable a chat
     */
    public function disable(Chat $chat)
    {
        $chat->update(['is_active' => false]);

        return redirect()->back()->with('success', 'Chat disabled successfully.');
    }

    /**
     * Enable a chat
     */
    public function enable(Chat $chat)
    {
        $chat->update(['is_active' => true]);

        return redirect()->back()->with('success', 'Chat enabled successfully.');
    }

    /**
     * Delete a chat
     */
    public function destroy(Chat $chat)
    {
        $chat->delete();

        return redirect()->route('admin.chats.index')->with('success', 'Chat deleted successfully.');
    }

    /**
     * Store a newly created chat
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'type' => 'required|in:direct,group,team',
            'name' => 'nullable|string|max:255',
            'participants' => 'required|array',
            'participants.*' => 'exists:users,id',
        ]);

        $chat = Chat::create([
            'type' => $validated['type'],
            'name' => $validated['name'] ?? null,
            'created_by' => auth()->id(),
        ]);

        $chat->participants()->attach($validated['participants']);

        return redirect()->route('admin.chats.show', $chat->id)
            ->with('success', 'Chat created successfully.');
    }

    /**
     * Post a message to a chat
     */
    public function postMessage(Chat $chat, Request $request)
    {
        $validated = $request->validate([
            'content' => 'required|string',
            'type' => 'nullable|in:text,image,file,system',
        ]);

        $message = $chat->messages()->create([
            'sender_id' => auth()->id(),
            'content' => $validated['content'],
            'message_type' => $validated['type'] ?? 'text',
        ]);

        return response()->json([
            'success' => true,
            'message' => $message,
        ]);
    }

    /**
     * Get messages for a chat
     */
    public function getMessages(Chat $chat, Request $request)
    {
        $page = $request->query('page', 1);
        $messages = $chat->messages()
            ->with('sender')
            ->latest()
            ->paginate(50);

        return response()->json([
            'data' => $messages->items(),
            'pagination' => [
                'total' => $messages->total(),
                'per_page' => $messages->perPage(),
                'current_page' => $messages->currentPage(),
                'last_page' => $messages->lastPage(),
            ],
        ]);
    }

    /**
     * Mark chat as read
     */
    public function markAsRead(Chat $chat, Request $request)
    {
        $chat->participants()
            ->where('user_id', auth()->id())
            ->update(['last_read_at' => now()]);

        return response()->json([
            'success' => true,
            'message' => 'Chat marked as read',
        ]);
    }
}

