<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AdminActivityLog;
use App\Models\AdminSession;
use App\Models\User;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    /**
     * Show admin dashboard with permission checks
     */
    public function index(Request $request)
    {
        // Check if user has admin role
        if (!auth()->user()->hasAnyRole(['super_admin', 'admin'])) {
            abort(403, 'Unauthorized. Admin access required.');
        }

        // Get statistics
        $stats = [
            'total_users' => User::count(),
            'active_users' => User::where('status', 'active')->count(),
            'suspended_users' => User::where('status', 'suspended')->count(),
            'active_sessions' => AdminSession::active()->count(),
            'today_activities' => AdminActivityLog::whereDate('created_at', today())->count(),
            'security_alerts' => AdminActivityLog::where('is_sensitive', true)
                ->where('created_at', '>=', now()->subHours(24))
                ->count(),
            'new_this_month' => User::whereMonth('created_at', now()->month)->count(),
            'new_this_week' => User::where('created_at', '>=', now()->subWeek())->count(),
        ];

        // Get recent activities (with permission check)
        $recentActivities = [];
        if (auth()->user()->hasPermissionTo('activities.view')) {
            $recentActivities = AdminActivityLog::with('adminUser')
                ->latest()
                ->limit(10)
                ->get();
        }

        // Get active sessions (with permission check)
        $activeSessions = [];
        if (auth()->user()->hasPermissionTo('sessions.view')) {
            $activeSessions = AdminSession::with('user')
                ->where('is_active', true)
                ->latest('last_activity_at')
                ->limit(5)
                ->get();
        }

        // Get top admins by activity (with permission check)
        $topAdmins = [];
        if (auth()->user()->hasPermissionTo('activities.view')) {
            $topAdmins = AdminActivityLog::selectRaw('admin_user_id, count(*) as count')
                ->where('created_at', '>=', now()->subDays(7))
                ->groupBy('admin_user_id')
                ->with('adminUser')
                ->orderBy('count', 'desc')
                ->limit(5)
                ->get();
        }

        return view('admin.dashboard', compact(
            'stats',
            'recentActivities',
            'activeSessions',
            'topAdmins'
        ));
    }

    /**
     * Show advanced dashboard
     */
    public function advancedDashboard()
    {
        return $this->index(request());
    }

    /**
     * Get dashboard statistics (API endpoint)
     */
    public function statistics()
    {
        if (!auth()->user()->hasAnyRole(['super_admin', 'admin'])) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $stats = [
            'total_users' => User::count(),
            'active_users' => User::where('status', 'active')->count(),
            'suspended_users' => User::where('status', 'suspended')->count(),
            'active_sessions' => AdminSession::active()->count(),
            'today_activities' => AdminActivityLog::whereDate('created_at', today())->count(),
            'this_week_activities' => AdminActivityLog::where('created_at', '>=', now()->subDays(7))->count(),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats,
        ]);
    }

    /**
     * Get dashboard activity summary
     */
    public function activitySummary()
    {
        if (!auth()->user()->hasPermissionTo('activities.view')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $summary = [
            'today' => AdminActivityLog::whereDate('created_at', today())->count(),
            'this_week' => AdminActivityLog::where('created_at', '>=', now()->subDays(7))->count(),
            'this_month' => AdminActivityLog::where('created_at', '>=', now()->startOfMonth())->count(),
            'sensitive' => AdminActivityLog::where('is_sensitive', true)->count(),
            'failed' => AdminActivityLog::where('status', 'failed')->count(),
            'by_action' => AdminActivityLog::selectRaw('action, count(*) as count')
                ->groupBy('action')
                ->limit(10)
                ->get(),
        ];

        return response()->json([
            'success' => true,
            'data' => $summary,
        ]);
    }
}

