<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\MarketingCampaign;
use App\Models\MarketingCampaignAnalytic;
use App\Models\MarketingPackage;
use Illuminate\Http\Request;

class MarketingController extends Controller
{
    /**
     * Dashboard - Overview of marketing
     */
    public function index()
    {
        $stats = [
            'total_packages' => MarketingPackage::count(),
            'active_packages' => MarketingPackage::where('is_active', true)->count(),
            'total_campaigns' => MarketingCampaign::count(),
            'active_campaigns' => MarketingCampaign::where('status', 'active')->count(),
            'pending_approval' => MarketingCampaign::where('status', 'pending')->count(),
            'total_revenue' => 0,
//            'total_revenue' => MarketingCampaign::whereIn('status', ['active', 'completed'])->sum('amount_paid'),
        ];

        $recentCampaigns = MarketingCampaign::with(['business', 'package'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('admin.marketing.index', compact('stats', 'recentCampaigns'));
    }

    /**
     * Manage packages
     */
    public function packages()
    {
        $packages = MarketingPackage::orderBy('type')
            ->orderBy('sort_order')
            ->get()
            ->groupBy('type');

        return view('admin.marketing.packages', compact('packages'));
    }

    /**
     * Create package
     */
    public function createPackage(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'type' => 'required|in:banner,featured_offer,featured_venue,popup,spotlight_boost',
            'duration_days' => 'required|integer|min:1',
            'price' => 'required|numeric|min:0',
            'features' => 'nullable|array',
            'max_impressions' => 'nullable|integer|min:1',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ]);

        $package = MarketingPackage::create($validated);

        AuditLog::logAction('marketing.package.created', $package);

        return response()->json([
            'success' => true,
            'message' => 'Package created successfully',
            'data' => $package,
        ]);
    }

    /**
     * Update package
     */
    public function updatePackage(Request $request, $id)
    {
        $package = MarketingPackage::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'duration_days' => 'required|integer|min:1',
            'price' => 'required|numeric|min:0',
            'features' => 'nullable|array',
            'max_impressions' => 'nullable|integer|min:1',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ]);

        $old = $package->toArray();
        $package->update($validated);

        AuditLog::logAction('marketing.package.updated', $package, $old, $package->toArray());

        return response()->json([
            'success' => true,
            'message' => 'Package updated successfully',
            'data' => $package,
        ]);
    }

    /**
     * Toggle package status
     */
    public function togglePackage($id)
    {
        $package = MarketingPackage::findOrFail($id);

        $package->update(['is_active' => !$package->is_active]);

        AuditLog::logAction('marketing.package.toggled', $package);

        return response()->json([
            'success' => true,
            'message' => 'Package status updated',
            'data' => $package,
        ]);
    }

    /**
     * Delete package
     */
    public function deletePackage($id)
    {
        $package = MarketingPackage::findOrFail($id);

        // Check if any campaigns use this package
        $campaignsCount = $package->campaigns()->whereIn('status', ['active', 'approved'])->count();

        if ($campaignsCount > 0) {
            return response()->json([
                'success' => false,
                'message' => "Cannot delete package with {$campaignsCount} active campaigns",
            ], 400);
        }

        AuditLog::logAction('marketing.package.deleted', $package, $package->toArray());

        $package->delete();

        return response()->json([
            'success' => true,
            'message' => 'Package deleted successfully',
        ]);
    }

    /**
     * All campaigns
     */
    public function campaigns(Request $request)
    {
        $query = MarketingCampaign::with(['business', 'package', 'venue', 'offer']);

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->type) {
            $query->where('type', $request->type);
        }

        if ($request->business_id) {
            $query->where('business_id', $request->business_id);
        }

        $campaigns = $query->orderBy('created_at', 'desc')->paginate(20);

        $stats = [
            'pending' => MarketingCampaign::where('status', 'pending')->count(),
            'active' => MarketingCampaign::where('status', 'active')->count(),
            'paused' => MarketingCampaign::where('status', 'paused')->count(),
            'completed' => MarketingCampaign::where('status', 'completed')->count(),
            'rejected' => MarketingCampaign::where('status', 'rejected')->count(),
        ];

        return view('admin.marketing.campaigns', compact('campaigns', 'stats'));
    }

    /**
     * View campaign details
     */
    public function viewCampaign($id)
    {
        $campaign = MarketingCampaign::with(['business', 'package', 'venue', 'offer', 'analytics', 'approver'])
            ->findOrFail($id);

        $aggregatedStats = MarketingCampaignAnalytic::getAggregatedStats($campaign->id);

        return view('admin.marketing.view-campaign', compact('campaign', 'aggregatedStats'));
    }

    /**
     * Approve campaign
     */
    public function approveCampaign($id, Request $request)
    {
        $campaign = MarketingCampaign::findOrFail($id);

        if ($campaign->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Campaign is not pending approval',
            ], 400);
        }

        $validated = $request->validate([
            'notes' => 'nullable|string|max:1000',
        ]);

        $campaign->update([
            'status' => 'approved',
            'approved_by' => auth()->id(),
            'approved_at' => now(),
        ]);

        // If start date is today or past, activate immediately
        if ($campaign->start_date->isPast() || $campaign->start_date->isToday()) {
            $campaign->update(['status' => 'active']);
        }

        AuditLog::logAction(
            'marketing.campaign.approved',
            $campaign,
            ['status' => 'pending'],
            ['status' => $campaign->status],
            ['notes' => $validated['notes'] ?? null]
        );

        // TODO: Send notification to business

        return response()->json([
            'success' => true,
            'message' => 'Campaign approved successfully',
            'data' => $campaign,
        ]);
    }

    /**
     * Reject campaign
     */
    public function rejectCampaign($id, Request $request)
    {
        $campaign = MarketingCampaign::findOrFail($id);

        if ($campaign->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Campaign is not pending approval',
            ], 400);
        }

        $validated = $request->validate([
            'reason' => 'required|string|max:1000',
        ]);

        $campaign->update([
            'status' => 'rejected',
            'rejection_reason' => $validated['reason'],
            'approved_by' => auth()->id(),
            'approved_at' => now(),
        ]);

        AuditLog::logAction(
            'marketing.campaign.rejected',
            $campaign,
            ['status' => 'pending'],
            ['status' => 'rejected', 'reason' => $validated['reason']]
        );

        // TODO: Send notification to business

        return response()->json([
            'success' => true,
            'message' => 'Campaign rejected',
        ]);
    }

    /**
     * Deactivate campaign
     */
    public function deactivateCampaign($id, Request $request)
    {
        $campaign = MarketingCampaign::findOrFail($id);

        $validated = $request->validate([
            'reason' => 'required|string|max:1000',
        ]);

        $campaign->update([
            'status' => 'paused',
            'rejection_reason' => $validated['reason'],
        ]);

        AuditLog::logAction(
            'marketing.campaign.deactivated',
            $campaign,
            ['status' => 'active'],
            ['status' => 'paused', 'reason' => $validated['reason']]
        );

        return response()->json([
            'success' => true,
            'message' => 'Campaign deactivated',
        ]);
    }

    /**
     * Get platform marketing analytics
     */
    public function analytics(Request $request)
    {
        $startDate = $request->start_date ?? now()->subDays(30);
        $endDate = $request->end_date ?? now();

        $totalCampaigns = MarketingCampaign::whereBetween('created_at', [$startDate, $endDate])->count();

        $totalRevenue = MarketingCampaign::whereBetween('created_at', [$startDate, $endDate])
            ->whereIn('status', ['active', 'completed'])
            ->sum('amount_paid');

        $analytics = MarketingCampaignAnalytic::whereBetween('date', [$startDate, $endDate])
            ->selectRaw('date, SUM(impressions) as impressions, SUM(clicks) as clicks, SUM(conversions) as conversions')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $campaignsByType = MarketingCampaign::whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('type, COUNT(*) as count, SUM(amount_paid) as revenue')
            ->groupBy('type')
            ->get();

        $topPerformers = MarketingCampaign::with('business')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->orderBy('clicks', 'desc')
            ->limit(10)
            ->get();

        return view('admin.marketing.analytics', compact(
            'analytics',
            'totalCampaigns',
            'totalRevenue',
            'campaignsByType',
            'topPerformers'
        ));
    }

    /**
     * Show create campaign form for admin
     */
    public function createCampaign()
    {
        $businesses = \App\Models\Business::where('status', 'verified')->get();
        $packages = MarketingPackage::where('is_active', true)->get()->groupBy('type');
        $venues = \App\Models\Venue::where('status', 'active')->get();
        $offers = \App\Models\Offer::where('status', 'active')->get();

        return view('admin.marketing.create-campaign', compact('businesses', 'packages', 'venues', 'offers'));
    }

    /**
     * Store new campaign from admin
     */
    public function storeCampaign(Request $request)
    {
        $validated = $request->validate([
            'business_id' => 'required|exists:businesses,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|string',
            'venue_id' => 'nullable|exists:venues,id',
            'offer_id' => 'nullable|exists:offers,id',
            'start_date' => 'required|date|after_or_equal:today',
            'end_date' => 'required|date|after:start_date',
            'budget' => 'required|numeric|min:0',
            'banner_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Handle banner image upload
        $bannerImagePath = null;
        if ($request->hasFile('banner_image')) {
            $file = $request->file('banner_image');
            $bannerImagePath = $file->store('banners', 'public');
        }

        $campaign = MarketingCampaign::create([
            'business_id' => $validated['business_id'],
            'name' => $validated['name'],
            'description' => $validated['description'],
            'type' => $validated['type'],
            'status' => 'approved',
            'approved_by' => auth()->id(),
            'approved_at' => now(),
            'start_date' => $validated['start_date'],
            'end_date' => $validated['end_date'],
            'budget' => $validated['budget'],
            'currency' => 'AED',
            'banner_image' => $bannerImagePath,
            'targeting_options' => [
                'venue_id' => $validated['venue_id'] ?? null,
                'offer_id' => $validated['offer_id'] ?? null,
            ],
            'goals' => [
                'impressions' => 0,
                'clicks' => 0,
                'conversions' => 0,
            ],
        ]);

        AuditLog::logAction('marketing.campaign.created', $campaign, [], $campaign->toArray());

        return response()->json([
            'success' => true,
            'message' => 'Campaign created successfully',
            'data' => $campaign,
        ]);
    }

    /**
     * Show edit campaign form
     */
    public function editCampaign($id)
    {
        $campaign = MarketingCampaign::findOrFail($id);
        $businesses = \App\Models\Business::where('status', 'approved')->get();
        $packages = MarketingPackage::where('is_active', true)->get()->groupBy('type');
        $venues = \App\Models\Venue::where('status', 'active')->get();
        $offers = \App\Models\Offer::where('status', 'active')->get();

        return view('admin.marketing.edit-campaign', compact('campaign', 'businesses', 'packages', 'venues', 'offers'));
    }

    /**
     * Update campaign
     */
    public function updateCampaign($id, Request $request)
    {
        $campaign = MarketingCampaign::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'venue_id' => 'nullable|exists:venues,id',
            'offer_id' => 'nullable|exists:offers,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'budget' => 'required|numeric|min:0',
        ]);

        $old = $campaign->toArray();

        $campaign->update([
            'name' => $validated['name'],
            'description' => $validated['description'],
            'start_date' => $validated['start_date'],
            'end_date' => $validated['end_date'],
            'budget' => $validated['budget'],
            'targeting_options' => [
                'venue_id' => $validated['venue_id'],
                'offer_id' => $validated['offer_id'],
            ],
        ]);

        AuditLog::logAction('marketing.campaign.updated', $campaign, $old, $campaign->toArray());

        return response()->json([
            'success' => true,
            'message' => 'Campaign updated successfully',
            'data' => $campaign,
        ]);
    }
}
