<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Role;
use App\Models\Permission;
use Illuminate\Http\Request;

class RoleController extends Controller
{
    /**
     * Display a listing of roles
     */
    public function index(Request $request)
    {
        $type = $request->query('type', 'all');

        $query = Role::withCount('permissions');

        // Filter by type if specified
        if ($type !== 'all' && in_array($type, ['admin', 'business', 'user'])) {
            $query->where('type', $type);
        }

        $roles = $query->paginate(50);

        // Calculate statistics
        $stats = [
            'total' => Role::count(),
            'admin' => Role::where('type', 'admin')->count(),
            'business' => Role::where('type', 'business')->count(),
            'system' => Role::where('is_system', true)->count(),
        ];

        return view('admin.roles.index', compact('roles', 'stats', 'type'));
    }

    /**
     * Show the form for creating a new role
     */
    public function create()
    {
        $permissions = Permission::all()->groupBy('group');

        return view('admin.roles.create', compact('permissions'));
    }

    /**
     * Store a newly created role
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|unique:roles|string|max:255',
            'display_name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500',
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role = Role::create($validated);

        if ($request->has('permissions')) {
            $role->permissions()->sync($request->permissions);
        }

        return redirect()->route('admin.roles.index')
            ->with('success', 'Role created successfully.');
    }

    /**
     * Display the specified role
     */
    public function show($id)
    {
        $role = Role::findOrFail($id);
        $role->load('permissions');

        return view('admin.roles.show', compact('role'));
    }

    /**
     * Show the form for editing the specified role
     */
    public function edit($id)
    {
        $role = Role::findOrFail($id);
        $role->load('permissions');
        $permissions = Permission::all()->groupBy('group');
        $rolePermissions = $role->permissions()->pluck('permissions.id')->toArray();

        return view('admin.roles.edit', compact('role', 'permissions', 'rolePermissions'));
    }

    /**
     * Update the specified role
     */
    public function update(Request $request, $id)
    {
        $role = Role::findOrFail($id);

        $validated = $request->validate([
            'display_name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500',
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role->update($validated);

        if ($request->has('permissions')) {
            $role->permissions()->sync($request->permissions);
        }

        return redirect()->route('admin.roles.index')
            ->with('success', 'Role updated successfully.');
    }

    /**
     * Delete the specified role
     */
    public function destroy($id)
    {
        $role = Role::findOrFail($id);

        // Prevent deletion of system roles
        if ($role->is_system) {
            return redirect()->route('admin.roles.index')
                ->with('error', 'System roles cannot be deleted.');
        }

        $role->delete();

        return redirect()->route('admin.roles.index')
            ->with('success', 'Role deleted successfully.');
    }

    /**
     * Display users with this role
     */
    public function users($id)
    {
        $role = Role::findOrFail($id);
        $role->load('users');
        $users = $role->users()->paginate(50);

        return view('admin.roles.users', compact('role', 'users'));
    }

    /**
     * Manage permissions for a role
     */
    public function managePermissions($id)
    {
        $role = Role::findOrFail($id);
        $role->load('permissions');
        $permissions = Permission::all()->groupBy('group');
        $rolePermissions = $role->permissions()->pluck('permissions.id')->toArray();

        return view('admin.roles.manage-permissions', compact('role', 'permissions', 'rolePermissions'));
    }

    /**
     * Show permissions for a role
     */
    public function permissions($id)
    {
        $role = Role::findOrFail($id);
        $role->load('permissions');
        $permissions = Permission::all()->groupBy('group');
        $rolePermissions = $role->permissions()->pluck('permissions.id')->toArray();

        return view('admin.roles.permissions', compact('role', 'permissions', 'rolePermissions'));
    }

    /**
     * Sync permissions for a role
     */
    public function syncPermissions(Request $request, $id)
    {
        $role = Role::findOrFail($id);

        $validated = $request->validate([
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        if ($request->has('permissions')) {
            $role->permissions()->sync($request->permissions);
        } else {
            $role->permissions()->detach();
        }

        return redirect()->route('admin.roles.permissions', $role->id)
            ->with('success', 'Permissions updated successfully.');
    }
}

