<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Spotlight;
use App\Models\Offer;
use Illuminate\Http\Request;

class SpotlightController extends Controller
{
    /**
     * Display a listing of spotlights
     */
    public function index(Request $request)
    {
        // Check permission - allow admin/super_admin or if they have spotlight.view permission
        $hasAccess = auth()->user()->hasAnyRole(['super_admin', 'admin'])
            || auth()->user()->hasPermissionTo('spotlight.view');

        if (!$hasAccess) {
            abort(403, 'Unauthorized');
        }

        $moderation = $request->query('moderation', 'all');
        $query = Spotlight::query();

        if ($moderation !== 'all') {
            $query->where('moderation_status', $moderation);
        }

        $spotlights = $query->with('user', 'offer')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $stats = [
            'total' => Spotlight::count(),
            'pending' => Spotlight::where('moderation_status', 'pending')->count(),
            'approved' => Spotlight::where('moderation_status', 'approved')->count(),
            'rejected' => Spotlight::where('moderation_status', 'rejected')->count(),
        ];

        return view('admin.spotlight.index', compact('spotlights', 'stats', 'moderation'));
    }

    /**
     * Show the form for creating a new spotlight for an offer
     */
    public function create(Request $request)
    {
        $hasAccess = auth()->user()->hasAnyRole(['super_admin', 'admin'])
            || auth()->user()->hasPermissionTo('spotlight.create');

        if (!$hasAccess) {
            abort(403, 'Unauthorized');
        }

        $offerId = $request->query('offer_id');
        $offer = $offerId ? Offer::findOrFail($offerId) : null;
        $offers = Offer::where('status', 'active')->get();

        return view('admin.spotlight.create', compact('offer', 'offers'));
    }

    /**
     * Store a newly created spotlight
     */
    public function store(Request $request)
    {
        $hasAccess = auth()->user()->hasAnyRole(['super_admin', 'admin'])
            || auth()->user()->hasPermissionTo('spotlight.create');

        if (!$hasAccess) {
            abort(403, 'Unauthorized');
        }

        $validated = $request->validate([
            'offer_id' => 'required|exists:offers,id',
            'content' => 'required|string',
            'media_type' => 'nullable|in:image,video,mixed',
            'media_urls' => 'nullable|array',
            'location' => 'nullable|string|max:255',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
        ]);

        $validated['user_id'] = auth()->id();
        $validated['moderation_status'] = 'pending';
        $validated['status'] = 'pending';

        Spotlight::create($validated);

        return redirect()->route('admin.spotlight.index')
            ->with('success', 'Spotlight created successfully and is pending moderation.');
    }

    /**
     * Display the specified spotlight
     */
    public function show($id)
    {
        $hasAccess = auth()->user()->hasAnyRole(['super_admin', 'admin'])
            || auth()->user()->hasPermissionTo('spotlight.view');

        if (!$hasAccess) {
            abort(403, 'Unauthorized');
        }

        $spotlight = Spotlight::with('user', 'offer', 'likes', 'comments')->findOrFail($id);

        return view('admin.spotlight.show', compact('spotlight'));
    }

    /**
     * Show the form for editing the specified spotlight
     */
    public function edit($id)
    {
        $hasAccess = auth()->user()->hasAnyRole(['super_admin', 'admin'])
            || auth()->user()->hasPermissionTo('spotlight.edit');

        if (!$hasAccess) {
            abort(403, 'Unauthorized');
        }

        $spotlight = Spotlight::findOrFail($id);
        $offers = Offer::where('status', 'active')->get();

        return view('admin.spotlight.edit', compact('spotlight', 'offers'));
    }

    /**
     * Update the specified spotlight
     */
    public function update(Request $request, $id)
    {
        $hasAccess = auth()->user()->hasAnyRole(['super_admin', 'admin'])
            || auth()->user()->hasPermissionTo('spotlight.edit');

        if (!$hasAccess) {
            abort(403, 'Unauthorized');
        }

        $spotlight = Spotlight::findOrFail($id);

        $validated = $request->validate([
            'offer_id' => 'required|exists:offers,id',
            'content' => 'required|string',
            'media_type' => 'nullable|in:image,video,mixed',
            'media_urls' => 'nullable|array',
            'location' => 'nullable|string|max:255',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
        ]);

        $spotlight->update($validated);

        return redirect()->route('admin.spotlight.show', $spotlight->id)
            ->with('success', 'Spotlight updated successfully.');
    }

    /**
     * Delete the specified spotlight
     */
    public function destroy($id)
    {
        $hasAccess = auth()->user()->hasAnyRole(['super_admin', 'admin'])
            || auth()->user()->hasPermissionTo('spotlight.delete');

        if (!$hasAccess) {
            abort(403, 'Unauthorized');
        }

        $spotlight = Spotlight::findOrFail($id);
        $spotlight->delete();

        return redirect()->route('admin.spotlight.index')
            ->with('success', 'Spotlight deleted successfully.');
    }

    /**
     * Approve a spotlight
     */
    public function approve($id)
    {
        $hasAccess = auth()->user()->hasAnyRole(['super_admin', 'admin'])
            || auth()->user()->hasPermissionTo('spotlight.approve');

        if (!$hasAccess) {
            abort(403, 'Unauthorized');
        }

        $spotlight = Spotlight::findOrFail($id);
        $spotlight->approve(auth()->id());

        return redirect()->route('admin.spotlight.show', $spotlight->id)
            ->with('success', 'Spotlight approved successfully.');
    }

    /**
     * Reject a spotlight
     */
    public function reject(Request $request, $id)
    {
        $hasAccess = auth()->user()->hasAnyRole(['super_admin', 'admin'])
            || auth()->user()->hasPermissionTo('spotlight.approve');

        if (!$hasAccess) {
            abort(403, 'Unauthorized');
        }

        $spotlight = Spotlight::findOrFail($id);
        $notes = $request->input('moderation_notes');
        $spotlight->reject(auth()->id(), $notes);

        return redirect()->route('admin.spotlight.show', $spotlight->id)
            ->with('success', 'Spotlight rejected successfully.');
    }

    /**
     * Get spotlights for a specific offer
     */
    public function forOffer($offerId)
    {
        $hasAccess = auth()->user()->hasAnyRole(['super_admin', 'admin'])
            || auth()->user()->hasPermissionTo('spotlight.view');

        if (!$hasAccess) {
            abort(403, 'Unauthorized');
        }

        $offer = Offer::findOrFail($offerId);
        $spotlights = $offer->spotlights()
            ->with('user', 'likes', 'comments')
            ->paginate(20);

        return view('admin.spotlight.offer-spotlights', compact('offer', 'spotlights'));
    }
}


