<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SubscriptionPlan;
use Illuminate\Http\Request;

class SubscriptionPlanController extends Controller
{
    /**
     * Display subscription plans listing
     */
    public function index(Request $request)
    {
        $type = $request->get('type', 'business'); // Default to business plans

        $plans = SubscriptionPlan::where('type', $type)
            ->withCount(['subscriptions', 'activeSubscriptions'])
            ->orderBy('sort_order')
            ->get();

        $stats = [
            'total_plans' => SubscriptionPlan::where('type', $type)->count(),
            'active_plans' => SubscriptionPlan::where('type', $type)->where('is_active', true)->count(),
            'total_subscriptions' => \App\Models\Subscription::whereHas('plan', function($q) use ($type) {
                $q->where('type', $type);
            })->count(),
            'monthly_revenue' => \App\Models\Payment::where('status', 'completed')
                ->where('type', 'subscription')
                ->whereMonth('created_at', now()->month)
                ->sum('amount'),
        ];

        // Separate counts for business and user plans
        $businessPlansCount = SubscriptionPlan::where('type', 'business')->count();
        $userPlansCount = SubscriptionPlan::where('type', 'user')->count();

        return view('admin.subscription-plans.index', compact('plans', 'stats', 'type', 'businessPlansCount', 'userPlansCount'));
    }

    /**
     * Show create form
     */
    public function create()
    {
        return view('admin.subscription-plans.create');
    }

    /**
     * Store new subscription plan
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:subscription_plans,name',
            'name_ar' => 'nullable|string|max:255',
            'description' => 'nullable|string|max:1000',
            'description_ar' => 'nullable|string|max:1000',
            'type' => 'required|in:business,user',
            'price' => 'required|numeric|min:0|max:999999.99',
            'duration_days' => 'required|integer|min:1|max:365',
            'duration_type' => 'required|in:monthly,yearly,custom',
            'features' => 'required|array|min:1',
            'features.*' => 'string|max:255',
            'max_venues' => 'nullable|integer|min:1|max:999',
            'max_offers' => 'nullable|integer|min:1|max:999',
            'max_staff' => 'nullable|integer|min:1|max:999',
            'max_photos_per_venue' => 'nullable|integer|min:1|max:999',
            'monthly_campaigns' => 'nullable|integer|min:0|max:999',
            'analytics_access' => 'nullable|boolean',
            'priority_support' => 'nullable|boolean',
            'featured_listing' => 'nullable|boolean',
            'custom_branding' => 'nullable|boolean',
            'api_access' => 'nullable|boolean',
            'sort_order' => 'nullable|integer|min:0|max:999',
            'is_active' => 'nullable|boolean',
            'is_popular' => 'nullable|boolean',
        ], [
            // Name validations
            'name.required' => 'Plan name is required',
            'name.string' => 'Plan name must be text',
            'name.max' => 'Plan name cannot exceed 255 characters',
            'name.unique' => 'A plan with this name already exists. Please choose a different name',

            // Arabic name validations
            'name_ar.string' => 'Arabic plan name must be text',
            'name_ar.max' => 'Arabic plan name cannot exceed 255 characters',

            // Description validations
            'description.string' => 'Description must be text',
            'description.max' => 'Description cannot exceed 1000 characters',
            'description_ar.string' => 'Arabic description must be text',
            'description_ar.max' => 'Arabic description cannot exceed 1000 characters',

            // Type validation
            'type.required' => 'Please select a plan type (Business or User)',
            'type.in' => 'Invalid plan type. Choose from: Business or User',

            // Price validations
            'price.required' => 'Plan price is required',
            'price.numeric' => 'Price must be a valid number',
            'price.min' => 'Price cannot be negative',
            'price.max' => 'Price cannot exceed 999,999.99',

            // Duration validations
            'duration_days.required' => 'Duration is required',
            'duration_days.integer' => 'Duration must be a whole number',
            'duration_days.min' => 'Duration must be at least 1 day',
            'duration_days.max' => 'Duration cannot exceed 365 days',
            'duration_type.required' => 'Please select a duration type (Monthly, Yearly, or Custom)',
            'duration_type.in' => 'Invalid duration type. Choose from: Monthly, Yearly, or Custom',

            // Features validations
            'features.required' => 'Please add at least one feature',
            'features.array' => 'Features must be in array format',
            'features.min' => 'Please add at least one feature to the plan',
            'features.*.string' => 'Each feature must be text',
            'features.*.max' => 'Each feature cannot exceed 255 characters',

            // Limits validations
            'max_venues.integer' => 'Max venues must be a whole number',
            'max_venues.min' => 'Max venues must be at least 1',
            'max_venues.max' => 'Max venues cannot exceed 999',
            'max_offers.integer' => 'Max offers must be a whole number',
            'max_offers.min' => 'Max offers must be at least 1',
            'max_offers.max' => 'Max offers cannot exceed 999',
            'max_staff.integer' => 'Max staff must be a whole number',
            'max_staff.min' => 'Max staff must be at least 1',
            'max_staff.max' => 'Max staff cannot exceed 999',
            'max_photos_per_venue.integer' => 'Max photos must be a whole number',
            'max_photos_per_venue.min' => 'Max photos must be at least 1',
            'max_photos_per_venue.max' => 'Max photos cannot exceed 999',

            // Sort order validation
            'sort_order.integer' => 'Sort order must be a whole number',
            'sort_order.min' => 'Sort order cannot be negative',
            'sort_order.max' => 'Sort order cannot exceed 999',
        ]);

        try {
            // Filter out empty features
            $features = array_filter($validated['features'], fn($f) => !empty(trim($f)));

            if (empty($features)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Please add at least one feature',
                    'errors' => ['features' => ['Please add at least one non-empty feature']]
                ], 422);
            }

            $validated['features'] = json_encode(array_values($features));
            $validated['analytics_access'] = $request->boolean('analytics_access', false);
            $validated['priority_support'] = $request->boolean('priority_support', false);
            $validated['featured_listing'] = $request->boolean('featured_listing', false);
            $validated['custom_branding'] = $request->boolean('custom_branding', false);
            $validated['api_access'] = $request->boolean('api_access', false);
            $validated['is_popular'] = $request->boolean('is_popular', false);
            $validated['is_active'] = $request->boolean('is_active', true);
            $validated['currency'] = 'AED';

            $plan = SubscriptionPlan::create($validated);

            return response()->json([
                'success' => true,
                'message' => ucfirst($validated['type']) . ' subscription plan created successfully!',
                'data' => $plan
            ]);
        } catch (\Exception $e) {
            \Log::error('Subscription plan creation error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error creating subscription plan: ' . $e->getMessage(),
                'errors' => []
            ], 500);
        }
    }

    /**
     * Show edit form
     */
    public function edit($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);

        // If AJAX request, return JSON
        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'plan' => $plan
            ]);
        }

        // Otherwise return view
        return view('admin.subscription-plans.edit', compact('plan'));
    }

    /**
     * Get plan as JSON (for AJAX requests)
     */
    public function show($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);

        return response()->json([
            'success' => true,
            'plan' => $plan
        ]);
    }

    /**
     * Update subscription plan
     */
    public function update(Request $request, $id)
    {
        $plan = SubscriptionPlan::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:subscription_plans,name,' . $id,
            'name_ar' => 'nullable|string|max:255',
            'description' => 'nullable|string|max:1000',
            'description_ar' => 'nullable|string|max:1000',
            'price' => 'required|numeric|min:0|max:999999.99',
            'duration_days' => 'required|integer|min:1|max:365',
            'duration_type' => 'required|in:monthly,yearly,custom',
            'features' => 'required|array|min:1',
            'features.*' => 'string|max:255',
            'max_venues' => 'nullable|integer|min:1|max:999',
            'max_offers' => 'nullable|integer|min:1|max:999',
            'max_staff' => 'nullable|integer|min:1|max:999',
            'max_photos_per_venue' => 'nullable|integer|min:1|max:999',
            'monthly_campaigns' => 'nullable|integer|min:0|max:999',
            'analytics_access' => 'boolean',
            'priority_support' => 'boolean',
            'featured_listing' => 'boolean',
            'custom_branding' => 'boolean',
            'api_access' => 'boolean',
            'sort_order' => 'nullable|integer|min:0|max:999',
            'is_active' => 'boolean',
            'is_popular' => 'boolean',
        ], [
            // Name validations
            'name.required' => 'Plan name is required',
            'name.string' => 'Plan name must be text',
            'name.max' => 'Plan name cannot exceed 255 characters',
            'name.unique' => 'A plan with this name already exists. Please choose a different name',

            // Arabic name validations
            'name_ar.string' => 'Arabic plan name must be text',
            'name_ar.max' => 'Arabic plan name cannot exceed 255 characters',

            // Description validations
            'description.string' => 'Description must be text',
            'description.max' => 'Description cannot exceed 1000 characters',
            'description_ar.string' => 'Arabic description must be text',
            'description_ar.max' => 'Arabic description cannot exceed 1000 characters',

            // Price validations
            'price.required' => 'Plan price is required',
            'price.numeric' => 'Price must be a valid number',
            'price.min' => 'Price cannot be negative',
            'price.max' => 'Price cannot exceed 999,999.99',

            // Duration validations
            'duration_days.required' => 'Duration is required',
            'duration_days.integer' => 'Duration must be a whole number',
            'duration_days.min' => 'Duration must be at least 1 day',
            'duration_days.max' => 'Duration cannot exceed 365 days',
            'duration_type.required' => 'Please select a duration type (Monthly, Yearly, or Custom)',
            'duration_type.in' => 'Invalid duration type. Choose from: Monthly, Yearly, or Custom',

            // Features validations
            'features.required' => 'Please add at least one feature',
            'features.array' => 'Features must be in array format',
            'features.min' => 'Please add at least one feature to the plan',
            'features.*.string' => 'Each feature must be text',
            'features.*.max' => 'Each feature cannot exceed 255 characters',

            // Limits validations
            'max_venues.integer' => 'Max venues must be a whole number',
            'max_venues.min' => 'Max venues must be at least 1',
            'max_venues.max' => 'Max venues cannot exceed 999',
            'max_offers.integer' => 'Max offers must be a whole number',
            'max_offers.min' => 'Max offers must be at least 1',
            'max_offers.max' => 'Max offers cannot exceed 999',
            'max_staff.integer' => 'Max staff must be a whole number',
            'max_staff.min' => 'Max staff must be at least 1',
            'max_staff.max' => 'Max staff cannot exceed 999',
            'max_photos_per_venue.integer' => 'Max photos must be a whole number',
            'max_photos_per_venue.min' => 'Max photos must be at least 1',
            'max_photos_per_venue.max' => 'Max photos cannot exceed 999',

            // Sort order validation
            'sort_order.integer' => 'Sort order must be a whole number',
            'sort_order.min' => 'Sort order cannot be negative',
            'sort_order.max' => 'Sort order cannot exceed 999',
        ]);

        try {
            // Filter out empty features
            $features = array_filter($validated['features'], fn($f) => !empty(trim($f)));

            if (empty($features)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Please add at least one feature',
                    'errors' => ['features' => ['Please add at least one non-empty feature']]
                ], 422);
            }

            $validated['features'] = json_encode(array_values($features));

            $plan->update($validated);

            return response()->json([
                'success' => true,
                'message' => 'Subscription plan updated successfully!',
                'data' => $plan
            ]);
        } catch (\Exception $e) {
            \Log::error('Subscription plan update error', [
                'plan_id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error updating subscription plan: ' . $e->getMessage(),
                'errors' => []
            ], 500);
        }
    }

    /**
     * Toggle plan status
     */
    public function toggle($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);
        $plan->update(['is_active' => !$plan->is_active]);

        return response()->json([
            'success' => true,
            'message' => 'Plan status updated',
            'data' => $plan
        ]);
    }

    /**
     * Toggle popular badge
     */
    public function togglePopular($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);
        $plan->update(['is_popular' => !$plan->is_popular]);

        return response()->json([
            'success' => true,
            'message' => 'Popular badge updated',
            'data' => $plan
        ]);
    }

    /**
     * Delete subscription plan
     */
    public function destroy($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);

        // Check if plan has active subscriptions
        if ($plan->subscriptions()->where('status', 'active')->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete plan with active subscriptions'
            ], 400);
        }

        $plan->delete();

        return response()->json([
            'success' => true,
            'message' => 'Subscription plan deleted successfully'
        ]);
    }

    /**
     * View plan subscribers
     */
    public function subscribers($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);

        $subscriptions = $plan->subscriptions()
            ->with(['business', 'business.owner'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.subscription-plans.subscribers', compact('plan', 'subscriptions'));
    }
}
