<?php
namespace App\Http\Controllers\Admin;
use App\Http\Controllers\Controller;
use App\Models\SupportTicket;
use App\Models\SupportTicketReply;
use Illuminate\Http\Request;
class SupportController extends Controller
{
    // List Tickets
    public function index()
    {
        $tickets = SupportTicket::with('user')
            ->latest()
            ->paginate(20);

        return view('admin.tickets.index', compact('tickets'));
    }

    // Show Single Ticket
    public function show($id)
    {
        $ticket = SupportTicket::with(['user', 'replies.user'])
            ->findOrFail($id);

        return view('admin.tickets.show', compact('ticket'));
    }

    // Close Ticket
    public function close($id)
    {
        $ticket = SupportTicket::findOrFail($id);
        $ticket->update(['status' => 'closed']);

        // Send notification to user about ticket closure
        $channels = config_value('notifications.channels.support_ticket_closed', ['database', 'email']);
        notify($ticket->user, 'support_ticket_closed', [
            'ticket_number' => $ticket->ticket_number,
            'subject' => $ticket->subject,
            'action_url' => route('user.support.show', $ticket->id),
        ], $channels);

        return redirect()
            ->route('admin.tickets.show', $id)
            ->with('success', 'Ticket closed successfully.');
    }

    // Reopen Ticket
    public function reopen($id)
    {
        $ticket = SupportTicket::findOrFail($id);
        $ticket->update(['status' => 'open']);

        // Send notification to user about ticket reopening
        $channels = config_value('notifications.channels.support_ticket_reopened', ['database']);
        notify($ticket->user, 'support_ticket_reopened', [
            'ticket_number' => $ticket->ticket_number,
            'subject' => $ticket->subject,
            'action_url' => route('user.support.show', $ticket->id),
        ], $channels);

        return redirect()
            ->route('admin.tickets.show', $id)
            ->with('success', 'Ticket reopened successfully.');
    }

    // Add Reply
    public function reply(Request $request, $id)
    {
        $request->validate([
            'message' => 'required|string',
        ]);

        $ticket = SupportTicket::findOrFail($id);

        SupportTicketReply::create([
            'support_ticket_id' => $ticket->id,
            'user_id' => auth()->id(),
            'message' => $request->message,
        ]);

        // Send notification to user about support ticket reply
        $channels = config_value('notifications.channels.support_ticket_replied', ['database', 'email']);
        $admin = auth()->user();
        notify($ticket->user, 'support_ticket_replied', [
            'ticket_number' => $ticket->ticket_number,
            'subject' => $ticket->subject,
            'message_preview' => substr($request->message, 0, 50),
            'from_user' => $admin->first_name . ' ' . $admin->last_name,
            'action_url' => route('user.support.show', $ticket->id),
        ], $channels);

        return redirect()
            ->route('admin.tickets.show', $id)
            ->with('success', 'Reply added successfully.');
    }

    // Update Status
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:open,closed,pending,resolved'
        ]);

        $ticket = SupportTicket::findOrFail($id);
        $oldStatus = $ticket->status;
        $ticket->update(['status' => $request->status]);

        // Send notification about status change
        if ($oldStatus !== $request->status) {
            $channels = config_value('notifications.channels.support_ticket_status_changed', ['database']);
            notify($ticket->user, 'support_ticket_status_changed', [
                'ticket_number' => $ticket->ticket_number,
                'old_status' => $oldStatus,
                'new_status' => $request->status,
                'action_url' => route('user.support.show', $ticket->id),
            ], $channels);
        }

        return redirect()
            ->route('admin.tickets.show', $id)
            ->with('success', 'Ticket status updated successfully.');
    }

    // Assign Ticket to Admin
    public function assign(Request $request, $id)
    {
        $request->validate([
            'admin_id' => 'required|exists:users,id'
        ]);

        $ticket = SupportTicket::findOrFail($id);
        $ticket->update([
            'assigned_admin_id' => $request->admin_id
        ]);

        // Send notification to assigned admin
        $channels = config_value('notifications.channels.support_ticket_assigned', ['database']);
        $assignedAdmin = \App\Models\User::find($request->admin_id);
        notify($assignedAdmin, 'support_ticket_assigned', [
            'ticket_number' => $ticket->ticket_number,
            'subject' => $ticket->subject,
            'priority' => $ticket->priority,
            'assigned_by' => auth()->user()->first_name . ' ' . auth()->user()->last_name,
            'action_url' => route('admin.tickets.show', $ticket->id),
        ], $channels);

        return redirect()
            ->route('admin.tickets.show', $id)
            ->with('success', 'Ticket assigned successfully.');
    }

    // Delete Ticket
    public function destroy($id)
    {
        $ticket = SupportTicket::findOrFail($id);
        $ticket->delete();

        return redirect()
            ->route('admin.tickets.index')
            ->with('success', 'Ticket deleted successfully.');
    }


}
