<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    /**
     * Display user listing
     */
    public function index(Request $request)
    {
        $query = User::query();

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('full_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by role
        if ($request->filled('role')) {
            $query->where('role', $request->role);
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $users = $query->paginate(15);

        // Stats
        $stats = [
            'total' => User::count(),
            'active' => User::where('status', 'active')->count(),
            'new_month' => User::whereMonth('created_at', now()->month)->count(),
            'suspended' => User::where('status', 'suspended')->count(),
        ];

        return view('admin.users.index', compact('users', 'stats'));
    }

    /**
     * Show user details
     */
    public function show($id)
    {
        $user = User::with(['business', 'subscriptions'])->findOrFail($id);
        return view('admin.users.show', compact('user'));
    }

    /**
     * Show user edit form
     */
    public function edit($id)
    {
        $user = User::findOrFail($id);
        return view('admin.users.edit', compact('user'));
    }

    /**
     * Update user
     */
    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);
        $oldStatus = $user->status;

        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $id,
            'phone' => 'required|string|unique:users,phone,' . $id,
            'role' => 'required|in:user,business_owner,admin',
            'status' => 'required|in:active,inactive,suspended',
        ]);

        $validated['full_name'] = $validated['first_name'] . ' ' . $validated['last_name'];

        $user->update($validated);

        // Send notification if status changed
        if ($oldStatus !== $validated['status']) {
            $channels = config_value('notifications.channels.user_status_changed', ['database', 'email']);

            $message = match($validated['status']) {
                'suspended' => 'Your account has been suspended',
                'inactive' => 'Your account has been deactivated',
                'active' => 'Your account has been reactivated',
                default => 'Your account status has been updated',
            };

            notify($user, 'user_status_changed', [
                'old_status' => $oldStatus,
                'new_status' => $validated['status'],
                'message' => $message,
                'action_url' => route('user.dashboard'),
            ], $channels);
        }

        return redirect()->route('admin.users.index')
            ->with('success', 'User updated successfully.');
    }

    /**
     * Delete user
     */
    public function destroy($id)
    {
        $user = User::findOrFail($id);

        // Prevent deleting yourself
        if ($user->id === auth()->id()) {
            return back()->with('error', 'You cannot delete your own account.');
        }

        $user->delete();

        return redirect()->route('admin.users.index')
            ->with('success', 'User deleted successfully.');
    }

    /**
     * Store new user
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'required|string|unique:users,phone',
            'password' => 'required|string|min:8',
            'role' => 'required|in:user,business_owner,admin',
        ]);

        User::create([
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'full_name' => $validated['first_name'] . ' ' . $validated['last_name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'password' => Hash::make($validated['password']),
            'role' => $validated['role'],
            'account_type' => $validated['role'] === 'business_owner' ? 'business_owner' : 'user',
            'status' => 'active',
            'email_verified_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'User created successfully'
        ]);
    }

    /**
     * Export users
     */
    public function export(Request $request)
    {
        $users = User::all();

        $csvData = [];
        $csvData[] = ['ID', 'Name', 'Email', 'Phone', 'Role', 'Status', 'Joined'];

        foreach ($users as $user) {
            $csvData[] = [
                $user->id,
                $user->full_name,
                $user->email,
                $user->phone,
                $user->role,
                $user->status,
                $user->created_at->format('Y-m-d'),
            ];
        }

        $filename = 'users_' . date('Y-m-d') . '.csv';

        $handle = fopen('php://temp', 'r+');
        foreach ($csvData as $row) {
            fputcsv($handle, $row);
        }
        rewind($handle);
        $csv = stream_get_contents($handle);
        fclose($handle);

        return response($csv)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    /**
     * Get user activity summary
     */
    public function getActivitySummary($id)
    {
        $user = User::findOrFail($id);
        $service = app(\App\Services\AdminUserManagementService::class);
        $summary = $service->getUserActivitySummary($user);

        return response()->json([
            'success' => true,
            'data' => $summary,
        ]);
    }

    /**
     * Get user permissions
     */
    public function getPermissions($id)
    {
        $user = User::findOrFail($id);
        $service = app(\App\Services\AdminUserManagementService::class);
        $permissions = $service->getUserPermissions($user);

        return response()->json([
            'success' => true,
            'data' => $permissions,
        ]);
    }

    /**
     * Export user data
     */
    public function exportData($id)
    {
        $user = User::findOrFail($id);
        $service = app(\App\Services\AdminUserManagementService::class);
        $data = $service->exportUserData($user);

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * Restore deleted user
     */
    public function restore($id)
    {
        $service = app(\App\Services\AdminUserManagementService::class);
        $user = $service->restoreUser($id, auth()->id());

        return redirect()->route('admin.users.index')
            ->with('success', "User {$user->email} restored successfully.");
    }

    /**
     * Bulk update user status
     */
    public function bulkUpdateStatus(Request $request)
    {
        $validated = $request->validate([
            'user_ids' => 'required|array|min:1',
            'user_ids.*' => 'integer|exists:users,id',
            'status' => 'required|in:active,suspended,inactive',
        ]);

        $service = app(\App\Services\AdminUserManagementService::class);
        $operation = $service->bulkUpdateStatus(
            $validated['user_ids'],
            $validated['status'],
            auth()->id()
        );

        // Send notifications to affected users
        $users = User::whereIn('id', $validated['user_ids'])->get();
        $channels = config_value('notifications.channels.user_status_changed', ['database', 'email']);

        foreach ($users as $user) {
            $message = match($validated['status']) {
                'suspended' => 'Your account has been suspended',
                'inactive' => 'Your account has been deactivated',
                'active' => 'Your account has been reactivated',
                default => 'Your account status has been updated',
            };

            notify($user, 'user_status_changed', [
                'new_status' => $validated['status'],
                'message' => $message,
                'action_url' => route('user.dashboard'),
            ], $channels);
        }

        return response()->json([
            'success' => true,
            'message' => 'Bulk operation initiated successfully',
            'operation_id' => $operation->id,
            'status' => $operation->status,
        ]);
    }

    /**
     * Bulk assign roles
     */
    public function bulkAssignRoles(Request $request)
    {
        $validated = $request->validate([
            'user_ids' => 'required|array|min:1',
            'user_ids.*' => 'integer|exists:users,id',
            'roles' => 'required|array|min:1',
            'roles.*' => 'integer|exists:roles,id',
        ]);

        $service = app(\App\Services\AdminUserManagementService::class);
        $operation = $service->bulkAssignRoles(
            $validated['user_ids'],
            $validated['roles'],
            auth()->id()
        );

        return response()->json([
            'success' => true,
            'message' => 'Bulk role assignment initiated',
            'operation_id' => $operation->id,
        ]);
    }

    /**
     * Bulk delete users
     */
    public function bulkDelete(Request $request)
    {
        $validated = $request->validate([
            'user_ids' => 'required|array|min:1',
            'user_ids.*' => 'integer|exists:users,id',
        ]);

        // Prevent deleting yourself
        if (in_array(auth()->id(), $validated['user_ids'])) {
            return response()->json([
                'success' => false,
                'message' => 'You cannot delete your own account.',
            ], 422);
        }

        $deletedCount = User::whereIn('id', $validated['user_ids'])->delete();

        \App\Models\AdminActivityLog::createFromRequest(
            auth()->id(),
            'bulk_delete',
            'User',
            null,
            'Multiple Users',
            description: "Bulk deleted {$deletedCount} users",
            isSensitive: true
        );

        return response()->json([
            'success' => true,
            'message' => "Successfully deleted {$deletedCount} users",
            'deleted_count' => $deletedCount,
        ]);
    }

    /**
     * Display user activity tracking
     */
    public function activity(Request $request)
    {
        $query = User::query();

        // Search by name or email
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Filter by account type
        if ($request->filled('account_type')) {
            $query->where('account_type', $request->account_type);
        }

        // Filter by status (based on activity)
        if ($request->filled('status')) {
            if ($request->status === 'active') {
                $query->where('last_active_at', '>=', now()->subDay());
            } elseif ($request->status === 'inactive') {
                $query->where(function($q) {
                    $q->where('last_active_at', '<', now()->subDay())
                      ->orWhereNull('last_active_at');
                });
            }
        }

        // Filter by activity days
        if ($request->filled('days')) {
            $days = (int)$request->days;
            $query->where('last_active_at', '>=', now()->subDays($days));
        }

        $users = $query->orderBy('last_active_at', 'desc')->paginate(50);

        // Calculate statistics
        $stats = [
            'total_users' => User::count(),
            'active_today' => User::where('last_active_at', '>=', now()->startOfDay())->count(),
            'active_this_week' => User::where('last_active_at', '>=', now()->subDays(7))->count(),
            'inactive_7_days' => User::where(function($q) {
                $q->where('last_active_at', '<', now()->subDays(7))
                  ->orWhereNull('last_active_at');
            })->count(),
        ];

        return view('admin.users.activity', compact('users', 'stats'));
    }
}
