<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Venue;
use Illuminate\Http\Request;

class VenueRejectionController extends Controller
{
    /**
     * Show venue rejection form
     */
    public function showRejectForm($id)
    {
        $venue = Venue::findOrFail($id);
        return view('admin.venues.reject', compact('venue'));
    }

    /**
     * Reject venue with reason and checklist
     */
    public function reject(Request $request, $id)
    {
        $venue = Venue::findOrFail($id);

        $validated = $request->validate([
            'reason' => 'required|string|min:10|max:1000',
            'checklist' => 'nullable|array',
            'checklist.*' => 'string',
        ]);

        // Build rejection checklist
        $checklist = [];
        $rejectionReasons = [
            'name_missing' => 'Venue Name Missing or Invalid',
            'address_missing' => 'Address Missing or Incomplete',
            'contact_invalid' => 'Contact Details Invalid',
            'description_insufficient' => 'Venue Description Insufficient',
            'images_missing' => 'Required Images Missing',
            'operating_hours' => 'Operating Hours Not Provided',
            'amenities_missing' => 'Amenities Not Specified',
            'compliance' => 'Venue Does Not Meet Compliance Requirements',
            'other' => 'Other Reason',
        ];

        foreach ($rejectionReasons as $key => $label) {
            $checklist[$key] = [
                'label' => $label,
                'checked' => in_array($key, $validated['checklist'] ?? []),
            ];
        }

        // Update venue status to rejected
        $venue->update([
            'status' => 'rejected',
            'rejection_reason' => $validated['reason'],
            'rejection_checklist' => $checklist,
            'verification_attempts' => ($venue->verification_attempts ?? 0) + 1,
        ]);

        // Send notification to business owner
        $business = $venue->business;
        $channels = config_value('notifications.channels.venue_rejected', ['database', 'email']);

        notify($business->owner, 'venue_rejected', [
            'venue_name' => $venue->name,
            'business_name' => $business->business_name,
            'reason' => $validated['reason'],
            'checklist' => $checklist,
            'action_url' => route('business.venues.show', $venue),
        ], $channels);

        return redirect()->route('admin.venues.index')
            ->with('success', 'Venue rejected successfully. Business owner has been notified.');
    }

    /**
     * Approve venue
     */
    public function approve($id)
    {
        $venue = Venue::findOrFail($id);

        $venue->update([
            'status' => 'active',
            'rejection_reason' => null,
            'rejection_checklist' => null,
        ]);

        // Send approval notification
        $business = $venue->business;
        $channels = config_value('notifications.channels.venue_approved', ['database', 'email']);

        notify($business->owner, 'venue_approved', [
            'venue_name' => $venue->name,
            'business_name' => $business->business_name,
            'action_url' => route('business.venues.show', $venue),
        ], $channels);

        return back()->with('success', 'Venue approved successfully.');
    }
}

