<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Models\Redemption;
use App\Models\Offer;
use App\Models\Venue;
use App\Models\Review;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AnalyticsController extends Controller
{
    /**
     * Display analytics dashboard
     */
    public function index()
    {
        $business = auth()->user()->business;

        // Overview stats
        $stats = [
            'total_venues' => $business->venues()->count(),
            'total_offers' => Offer::whereHas('venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->count(),
            'total_redemptions' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->count(),
            'total_reviews' => Review::whereHas('venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->count(),
            'average_rating' => Review::whereHas('venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->avg('rating') ?? 0,
            'total_views' => $business->venues()->sum('view_count'),
        ];

        // This month comparison
        $thisMonth = [
            'redemptions' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->whereMonth('created_at', now()->month)->count(),
            'reviews' => Review::whereHas('venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->whereMonth('created_at', now()->month)->count(),
        ];

        $lastMonth = [
            'redemptions' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->whereMonth('created_at', now()->subMonth()->month)->count(),
            'reviews' => Review::whereHas('venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->whereMonth('created_at', now()->subMonth()->month)->count(),
        ];

        // Top performing offers
        $topOffers = Offer::whereHas('venue', function($q) use ($business) {
            $q->where('business_id', $business->id);
        })
        ->withCount('redemptions')
        ->orderByDesc('redemptions_count')
        ->limit(5)
        ->get();

        // Recent redemptions trend (last 30 days)
        $redemptionTrend = Redemption::whereHas('offer.venue', function($q) use ($business) {
            $q->where('business_id', $business->id);
        })
        ->where('created_at', '>=', now()->subDays(30))
        ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
        ->groupBy('date')
        ->orderBy('date')
        ->get();

        return view('business.analytics.index', compact(
            'stats',
            'thisMonth',
            'lastMonth',
            'topOffers',
            'redemptionTrend'
        ));
    }

    /**
     * Offers analytics
     */
    public function offers(Request $request)
    {
        $business = auth()->user()->business;

        $query = Offer::whereHas('venue', function($q) use ($business) {
            $q->where('business_id', $business->id);
        })->withCount('redemptions');

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $offers = $query->orderByDesc('redemptions_count')->paginate(20);

        return view('business.analytics.offers', compact('offers'));
    }

    /**
     * Venues analytics
     */
    public function venues(Request $request)
    {
        $business = auth()->user()->business;

        $venues = $business->venues()
            ->withCount(['offers', 'reviews'])
            ->withAvg('reviews', 'rating')
            ->orderByDesc('view_count')
            ->paginate(20);

        return view('business.analytics.venues', compact('venues'));
    }

    /**
     * Revenue analytics
     */
    public function revenue()
    {
        $business = auth()->user()->business;

        $monthlyRevenue = DB::table('payments')
            ->where('business_id', $business->id)
            ->where('status', 'completed')
            ->where('created_at', '>=', now()->subYear())
            ->selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month, SUM(amount) as total')
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        $totalRevenue = DB::table('payments')
            ->where('business_id', $business->id)
            ->where('status', 'completed')
            ->sum('amount');

        return view('business.analytics.revenue', compact('monthlyRevenue', 'totalRevenue'));
    }

    /**
     * Customer analytics
     */
    public function customers()
    {
        $business = auth()->user()->business;

        // Unique customers who redeemed offers
        $customers = Redemption::whereHas('offer.venue', function($q) use ($business) {
            $q->where('business_id', $business->id);
        })
        ->select('user_id')
        ->selectRaw('COUNT(*) as redemption_count')
        ->groupBy('user_id')
        ->with('user')
        ->orderByDesc('redemption_count')
        ->paginate(20);

        $stats = [
            'total_customers' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->distinct('user_id')->count('user_id'),
            'new_this_month' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->whereMonth('created_at', now()->month)
              ->distinct('user_id')
              ->count('user_id'),
        ];

        return view('business.analytics.customers', compact('customers', 'stats'));
    }

    /**
     * Export analytics report
     */
    public function export(Request $request)
    {
        $business = auth()->user()->business;

        $data = [
            'business' => $business,
            'venues' => $business->venues()->withCount('offers', 'reviews')->get(),
            'offers' => Offer::whereHas('venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->withCount('redemptions')->get(),
            'redemptions' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->with(['offer', 'user'])->get(),
        ];

        // Generate CSV
        $csv = "Analytics Report for {$business->business_name}\n";
        $csv .= "Generated on: " . now()->format('Y-m-d H:i:s') . "\n\n";

        $csv .= "Venues,Offers,Redemptions\n";
        $csv .= "{$data['venues']->count()},{$data['offers']->count()},{$data['redemptions']->count()}\n";

        return response($csv)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="analytics-report.csv"');
    }
}
