<?php

namespace App\Http\Controllers\Business;

use App\Models\Chat;
use App\Models\Message;
use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class ChatController extends Controller
{
    /**
     * Display list of chats for business user
     */
    public function index(Request $request)
    {
        $user = auth()->user();

        // Get all chats for the user
        $chats = Chat::where(function ($query) use ($user) {
            $query->where('created_by', $user->id)
                  ->orWhereHas('participants', function ($q) use ($user) {
                      $q->where('user_id', $user->id);
                  });
        })
        ->with(['participants', 'messages' => function ($query) {
            $query->latest()->limit(1);
        }])
        ->latest('updated_at')
        ->paginate(20);

        return view('business.chats.index', compact('chats'));
    }

    /**
     * Show a specific chat
     */
    public function show(Chat $chat, Request $request)
    {
        // Check if user is part of this chat
        if ($chat->created_by !== auth()->id() && !$chat->participants()->where('user_id', auth()->id())->exists()) {
            abort(403, 'Unauthorized');
        }

        // Load messages with pagination
        $messages = $chat->messages()
            ->with('sender')
            ->latest()
            ->paginate(50);

        $messages = $messages->reverse();

        return view('business.chats.show', compact('chat', 'messages'));
    }

    /**
     * Create a new chat
     */
    public function create()
    {
        $user = auth()->user();
        $business = null;

        // Get user's business - could be owner
        if ($user->isBusinessOwner()) {
            $business = $user->business;
        }

        // If not owner, check if they're a staff member
        if (!$business) {
            $staffRecord = \App\Models\BusinessStaff::where('user_id', $user->id)
                ->where('status', 'active')
                ->first();

            if ($staffRecord) {
                $business = $staffRecord->business;
            }
        }

        if (!$business) {
            abort(403, 'User does not have access to any business');
        }

        // Get all other participants in the business:
        // - For owner: get all active staff
        // - For staff: get owner and other staff
        $participants = [];

        if ($user->isBusinessOwner()) {
            // Owner can chat with any active staff
            $participants = \App\Models\BusinessStaff::where('business_id', $business->id)
                ->where('status', 'active')
                ->with('user')
                ->get()
                ->filter(function ($staff) use ($user) {
                    return $staff->user_id && $staff->user_id !== $user->id && $staff->user;
                })
                ->map(function ($staff) {
                    return [
                        'id' => $staff->user_id,
                        'name' => $staff->user->first_name . ' ' . $staff->user->last_name,
                        'email' => $staff->user->email,
                        'role' => $staff->role,
                        'type' => 'staff',
                    ];
                })
                ->values();
        } else {
            // Staff can chat with the owner
            if ($business->owner) {
                $participants[] = [
                    'id' => $business->owner->id,
                    'name' => $business->owner->first_name . ' ' . $business->owner->last_name,
                    'email' => $business->owner->email,
                    'role' => 'owner',
                    'type' => 'owner',
                ];
            }

            // Staff can also chat with other staff
            $otherStaff = \App\Models\BusinessStaff::where('business_id', $business->id)
                ->where('status', 'active')
                ->where('user_id', '!=', $user->id)
                ->with('user')
                ->get()
                ->filter(function ($staff) {
                    return $staff->user_id && $staff->user;
                })
                ->map(function ($staff) {
                    return [
                        'id' => $staff->user_id,
                        'name' => $staff->user->first_name . ' ' . $staff->user->last_name,
                        'email' => $staff->user->email,
                        'role' => $staff->role,
                        'type' => 'staff',
                    ];
                })
                ->values();

            $participants = array_merge($participants, $otherStaff->toArray());
        }

        return view('business.chats.create', compact('business', 'participants'));
    }

    /**
     * Store a new chat
     */
    public function store(Request $request)
    {
        $user = auth()->user();

        // Get user's business - could be owner or staff
        $business = null;

        if ($user->isBusinessOwner()) {
            $business = $user->business;
        } else {
            $staffRecord = \App\Models\BusinessStaff::where('user_id', $user->id)
                ->where('status', 'active')
                ->first();

            if ($staffRecord) {
                $business = $staffRecord->business;
            }
        }

        if (!$business) {
            abort(403, 'User does not have access to any business');
        }

        $validated = $request->validate([
            'participant_ids' => 'required|array',
            'participant_ids.*' => 'integer|exists:users,id',
            'subject' => 'nullable|string|max:255',
            'message' => 'nullable|string',
        ]);

        // Generate chat name from participants
        $participantNames = User::whereIn('id', $validated['participant_ids'])
            ->pluck('first_name')
            ->join(', ');

        $chatName = count($validated['participant_ids']) > 1
            ? $validated['subject'] ?? "Group Chat: {$participantNames}"
            : $participantNames;

        // Create chat with required fields
        $chat = Chat::create([
            'created_by' => auth()->id(),
            'business_id' => $business->id,  // Set business_id
            'name' => $chatName,  // Set name (required)
            'type' => count($validated['participant_ids']) > 1 ? 'group' : 'direct',
            'is_active' => true,
        ]);

        // Add participants
        $participantIds = $validated['participant_ids'];
        $participantIds[] = auth()->id(); // Add creator
        $participantIds = array_unique($participantIds);

        // Attach participants with proper pivot data
        $pivotData = [];
        foreach ($participantIds as $id) {
            $pivotData[$id] = [
                'joined_at' => now(),
                'last_read_at' => now(),
            ];
        }
        $chat->participants()->attach($pivotData);

        // Send first message if provided
        if (!empty($validated['message'])) {
            Message::create([
                'chat_id' => $chat->id,
                'sender_id' => auth()->id(),
                'content' => $validated['message'],
                'message_type' => 'text',
            ]);
        }

        // Send notifications to participants about new chat
        $channels = config_value('notifications.channels.chat_created', ['database']);
        $currentUser = auth()->user();

        foreach ($validated['participant_ids'] as $participantId) {
            if ($participantId !== auth()->id()) {
                $participant = User::find($participantId);
                notify($participant, 'chat_created', [
                    'chat_name' => $chatName,
                    'initiated_by' => $currentUser->first_name . ' ' . $currentUser->last_name,
                    'action_url' => route('business.chats.show', $chat->id),
                ], $channels);
            }
        }

        return redirect()->route('business.chats.show', $chat->id)
            ->with('success', 'Chat created successfully.');
    }

    /**
     * Send a message in a chat
     */
    public function sendMessage(Request $request, Chat $chat)
    {
        // Check authorization
        if ($chat->created_by !== auth()->id() && !$chat->participants()->where('user_id', auth()->id())->exists()) {
            abort(403, 'Unauthorized');
        }

        // Check if chat is disabled
        if (!$chat->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'This chat is disabled and cannot receive messages.',
            ], 403);
        }

        $validated = $request->validate([
            'message' => 'required|string',
        ]);

        // Create message
        $message = Message::create([
            'chat_id' => $chat->id,
            'sender_id' => auth()->id(),
            'content' => $validated['message'],
            'message_type' => 'text',
        ]);

        // Update chat timestamp
        $chat->touch();

        // Send message notifications to other participants
        $messageChannels = config_value('notifications.channels.new_message', ['database']);
        $sender = auth()->user();

        foreach ($chat->participants as $participant) {
            if ($participant->id !== auth()->id()) {
                notify($participant, 'new_message', [
                    'from_user' => $sender->first_name . ' ' . $sender->last_name,
                    'chat_name' => $chat->name,
                    'message_preview' => substr($validated['message'], 0, 50),
                    'action_url' => route('business.chats.show', $chat->id),
                ], $messageChannels);
            }
        }

        // If AJAX request, return JSON
        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => $message,
                'sender' => $message->sender,
            ]);
        }

        return redirect()->route('business.chats.show', $chat->id);
    }

    /**
     * Delete a chat
     */
    public function destroy(Chat $chat)
    {
        // Check authorization
        if ($chat->created_by !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        $chat->delete();

        return redirect()->route('business.chats.index')
            ->with('success', 'Chat deleted successfully');
    }

    /**
     * Mark messages as read
     */
    public function markAsRead(Chat $chat)
    {
        $chat->participants()
            ->where('user_id', auth()->id())
            ->update(['last_read_at' => now()]);

        return response()->json(['success' => true]);
    }
}

