<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Services\NotificationService;
use Illuminate\Http\Request;

class NotificationController extends Controller
{
    /**
     * Display all notifications
     */
    public function index()
    {
        $user = auth()->user();

        // Get all notifications
        $allNotifications = NotificationService::getAllNotifications($user);
        $unreadNotifications = NotificationService::getUnreadNotifications($user);
        $unreadCount = $unreadNotifications->count();

        return view('business.notifications.index', [
            'allNotifications' => $allNotifications,
            'unreadNotifications' => $unreadNotifications,
            'unreadCount' => $unreadCount,
        ]);
    }

    /**
     * Mark single notification as read
     */
    public function markAsRead($id)
    {
        $notification = NotificationService::markAsRead($id);

        if (request()->wantsJson()) {
            return response()->json(['success' => true]);
        }

        return back()->with('success', 'Notification marked as read');
    }

    /**
     * Mark all notifications as read
     */
    public function markAllAsRead()
    {
        $user = auth()->user();
        NotificationService::markAllAsRead($user);

        if (request()->wantsJson()) {
            return response()->json(['success' => true]);
        }

        return back()->with('success', 'All notifications marked as read');
    }

    /**
     * Delete a notification
     */
    public function destroy($id)
    {
        $notification = \App\Models\Notification::find($id);

        if ($notification) {
            $notification->delete();
        }

        if (request()->wantsJson()) {
            return response()->json(['success' => true]);
        }

        return back()->with('success', 'Notification deleted');
    }

    /**
     * Get notification title based on type
     */
    public static function getNotificationTitle($notification)
    {
        $titles = [
            'offer_redeemed' => 'Offer Redeemed',
            'offer_created' => 'New Offer Created',
            'offer_expired' => 'Offer Expired',
            'staff_invited' => 'Staff Member Invited',
            'staff_joined' => 'Staff Member Joined',
            'review_received' => 'New Review Received',
            'system_alert' => 'System Alert',
            'venue_created' => 'Venue Created',
            'subscription_expiring' => 'Subscription Expiring Soon',
        ];

        return $titles[$notification->type] ?? ucwords(str_replace('_', ' ', $notification->type));
    }

    /**
     * Get notification message based on type
     */
    public static function getNotificationMessage($notification)
    {
        $data = $notification->data;

        switch ($notification->type) {
            case 'offer_redeemed':
                return "Your offer '{$data['offer_name']}' was redeemed by {$data['user_name']}";
            case 'offer_created':
                return "New offer '{$data['offer_name']}' has been created";
            case 'offer_expired':
                return "Your offer '{$data['offer_name']}' has expired";
            case 'staff_invited':
                return "{$data['staff_name']} has been invited to join your team";
            case 'staff_joined':
                return "{$data['staff_name']} has joined your team";
            case 'review_received':
                return "New review from {$data['reviewer_name']}: {$data['rating']} stars";
            case 'system_alert':
                return $data['message'] ?? 'System notification';
            case 'venue_created':
                return "New venue '{$data['venue_name']}' has been created";
            case 'subscription_expiring':
                return "Your subscription expires on {$data['expiry_date']}";
            default:
                return $data['message'] ?? 'You have a new notification';
        }
    }
}

