<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Models\Redemption;
use App\Models\Offer;
use App\Models\User;
use App\Traits\GetUserBusiness;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class RedemptionController extends Controller
{
    use GetUserBusiness;
    /**
     * Display redemptions listing
     */
    public function index(Request $request)
    {
        // Get business - handle both owner and staff
        if (auth()->user()->hasRole('business_manager') || auth()->user()->account_type === 'user') {
            $business = auth()->user()->getOwnerBusinessForStaff();
        } else {
            $business = auth()->user()->business;
        }

        if (!$business) {
            return redirect()->route('business.register')
                ->with('error', 'Please complete business registration first.');
        }

        $query = Redemption::whereHas('offer.venue', function($q) use ($business) {
            $q->where('business_id', $business->id);
        })
            ->with(['offer', 'user', 'venue']);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('redeemed_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('redeemed_at', '<=', $request->date_to);
        }

        if ($request->filled('offer_id')) {
            $query->where('offer_id', $request->offer_id);
        }

        $redemptions = $query->orderBy('created_at', 'desc')->paginate(20);

        $stats = [
            'total' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->count(),
            'today' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->whereDate('redeemed_at', today())->count(),
            'this_week' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->whereBetween('redeemed_at', [now()->startOfWeek(), now()->endOfWeek()])->count(),
            'this_month' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->whereMonth('redeemed_at', now()->month)->count(),
        ];

        $offers = Offer::whereHas('venue', function($q) use ($business) {
            $q->where('business_id', $business->id);
        })->where('status', 'active')->get();

        return view('business.redemptions.index', compact('redemptions', 'stats', 'offers'));
    }

    /**
     * Show QR code scanner
     */
    public function scan()
    {
        return view('business.redemptions.scan');
    }

    /**
     * Validate redemption code
     */
    public function validate(Request $request)
    {
        $request->validate([
            'code' => 'required|string'
        ]);

        $redemption = Redemption::where('redemption_code', $request->code)
            ->with(['offer', 'user'])
            ->first();

        if (!$redemption) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid redemption code'
            ], 404);
        }

        if ($redemption->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'This code has already been used'
            ], 400);
        }

        if ($redemption->expires_at && $redemption->expires_at->isPast()) {
            return response()->json([
                'success' => false,
                'message' => 'This redemption code has expired'
            ], 400);
        }

        return response()->json([
            'success' => true,
            'redemption' => $redemption,
            'offer' => $redemption->offer,
            'user' => $redemption->user
        ]);
    }

    /**
     * Process redemption
     */
    public function redeem(Request $request)
    {
        $request->validate([
            'code' => 'required|string'
        ]);

        $business = auth()->user()->business;

        $redemption = Redemption::where('redemption_code', $request->code)
            ->whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })
            ->first();

        if (!$redemption) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid redemption code or not authorized'
            ], 404);
        }

        if ($redemption->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'This code has already been redeemed'
            ], 400);
        }

        if ($redemption->expires_at && $redemption->expires_at->isPast()) {
            $redemption->update(['status' => 'expired']);
            return response()->json([
                'success' => false,
                'message' => 'This redemption code has expired'
            ], 400);
        }

        $redemption->update([
            'status' => 'redeemed',
            'redeemed_at' => now(),
            'redeemed_by' => auth()->id()
        ]);

        // Update offer redemption count
        $redemption->offer->increment('redemption_count');

        // Send notification to user
        $channels = config_value('notifications.channels.offer_redeemed', ['database']);
        notify($redemption->user, 'offer_redeemed', [
            'offer_name' => $redemption->offer->title,
            'business_name' => $redemption->offer->venue->business->business_name,
            'redeemed_at' => now()->format('Y-m-d H:i'),
            'action_url' => route('user.offers.show', $redemption->offer),
        ], $channels);

        return response()->json([
            'success' => true,
            'message' => 'Offer redeemed successfully!',
            'redemption' => $redemption
        ]);
    }

    /**
     * Show redemption statistics
     */
    public function statistics()
    {
        $business = auth()->user()->business;

        $stats = [
            'total_redemptions' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->count(),

            'by_status' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->selectRaw('status, COUNT(*) as count')
                ->groupBy('status')
                ->pluck('count', 'status'),

            'top_offers' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })
                ->selectRaw('offer_id, COUNT(*) as count')
                ->groupBy('offer_id')
                ->orderByDesc('count')
                ->limit(5)
                ->with('offer')
                ->get(),

            'daily_redemptions' => Redemption::whereHas('offer.venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })
                ->where('redeemed_at', '>=', now()->subDays(30))
                ->selectRaw('DATE(redeemed_at) as date, COUNT(*) as count')
                ->groupBy('date')
                ->orderBy('date')
                ->get(),
        ];

        return view('business.redemptions.statistics', compact('stats'));
    }

    /**
     * Show redemption details
     */
    public function show($id)
    {
        $business = auth()->user()->business;

        $redemption = Redemption::whereHas('offer.venue', function($q) use ($business) {
            $q->where('business_id', $business->id);
        })
            ->with(['offer', 'user', 'venue'])
            ->findOrFail($id);

        return view('business.redemptions.show', compact('redemption'));
    }

    /**
     * Verify redemption
     */
    public function verify(Request $request, $id)
    {
        $business = auth()->user()->business;

        $redemption = Redemption::whereHas('offer.venue', function($q) use ($business) {
            $q->where('business_id', $business->id);
        })->findOrFail($id);

        if ($redemption->status !== 'pending') {
            return back()->with('error', 'This redemption has already been processed.');
        }

        $redemption->update([
            'status' => 'redeemed',
            'redeemed_at' => now(),
            'redeemed_by' => auth()->id(),
        ]);

        // Send notification to user
        $channels = config_value('notifications.channels.offer_redeemed', ['database']);
        notify($redemption->user, 'offer_redeemed', [
            'offer_name' => $redemption->offer->title,
            'business_name' => $redemption->offer->venue->business->business_name,
            'redeemed_at' => now()->format('Y-m-d H:i'),
            'action_url' => route('user.offers.show', $redemption->offer),
        ], $channels);

        return back()->with('success', 'Redemption verified successfully!');
    }

    /**
     * Reject redemption
     */
    public function reject(Request $request, $id)
    {
        $business = auth()->user()->business;

        $redemption = Redemption::whereHas('offer.venue', function($q) use ($business) {
            $q->where('business_id', $business->id);
        })->findOrFail($id);

        if ($redemption->status !== 'pending') {
            return back()->with('error', 'This redemption has already been processed.');
        }

        $redemption->update([
            'status' => 'rejected',
            'rejected_at' => now(),
            'rejected_by' => auth()->id(),
            'rejection_reason' => $request->reason,
        ]);

        // Send notification to user about rejection
        $channels = config_value('notifications.channels.offer_rejected', ['database', 'email']);
        notify($redemption->user, 'offer_rejected', [
            'offer_name' => $redemption->offer->title,
            'business_name' => $redemption->offer->venue->business->business_name,
            'reason' => $request->reason ?? 'No reason provided',
            'action_url' => route('user.offers.show', $redemption->offer),
        ], $channels);

        return back()->with('success', 'Redemption rejected.');
    }

    /**
     * Export redemptions
     */
    public function export(Request $request)
    {
        $business = auth()->user()->business;

        $redemptions = Redemption::whereHas('offer.venue', function($q) use ($business) {
            $q->where('business_id', $business->id);
        })
            ->with(['offer', 'user', 'venue'])
            ->get();

        $csv = "ID,Offer,User,Venue,Status,Redeemed At,Created At\n";

        foreach ($redemptions as $redemption) {
            $csv .= sprintf(
                "%d,%s,%s,%s,%s,%s,%s\n",
                $redemption->id,
                $redemption->offer->title,
                $redemption->user->full_name,
                $redemption->venue->name,
                $redemption->status,
                $redemption->redeemed_at ?? 'N/A',
                $redemption->created_at
            );
        }

        return response($csv)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="redemptions.csv"');
    }
}
