<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\User;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;

class RegistrationController extends Controller
{
    /**
     * Show business registration form
     */
    public function showRegistrationForm()
    {
        return view('business.auth.register');
    }

    /**
     * Handle business registration
     */
    public function register(Request $request)
    {
        $validated = $request->validate([
            // Owner Information
            'owner_first_name' => 'required|string|max:255',
            'owner_last_name' => 'required|string|max:255',
            'owner_email' => 'required|email|unique:users,email',
            'owner_phone' => 'required|string|unique:users,phone',
            'password' => 'required|string|min:8|confirmed',

            // Business Information
            'business_name' => 'required|string|max:255',
            'business_name_ar' => 'nullable|string|max:255',
            'trade_license_number' => 'required|string|unique:businesses,trade_license_number',
            'trade_license_document' => 'required|file|mimes:pdf,jpg,jpeg,png|max:5120',
            'email' => 'required|email',
            'phone' => 'required|string',
            'website' => 'nullable|url',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'cover_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        DB::beginTransaction();

        try {
            // Create owner account
            $user = User::create([
                'first_name' => $validated['owner_first_name'],
                'last_name' => $validated['owner_last_name'],
                'full_name' => $validated['owner_first_name'] . ' ' . $validated['owner_last_name'],
                'email' => $validated['owner_email'],
                'phone' => $validated['owner_phone'],
                'password' => Hash::make($validated['password']),
                'account_type' => 'business',
                'role' => 'business_owner',
                'status' => 'active',
                'email_verified_at' => now(),
            ]);

            // Handle file uploads
            $tradeLicensePath = $request->file('trade_license_document')
                ->store('licenses', 'public');

            $logoPath = $request->hasFile('logo')
                ? $request->file('logo')->store('logos', 'public')
                : null;

            $coverPath = $request->hasFile('cover_image')
                ? $request->file('cover_image')->store('covers', 'public')
                : null;

            // Create business
            $business = Business::create([
                'owner_id' => $user->id,
                'business_name' => $validated['business_name'],
                'business_name_ar' => $validated['business_name_ar'] ?? null,
                'trade_license_number' => $validated['trade_license_number'],
                'trade_license_document' => $tradeLicensePath,
                'email' => $validated['email'],
                'phone' => $validated['phone'],
                'website' => $validated['website'] ?? null,
                'description' => $validated['description'] ?? null,
                'description_ar' => $validated['description_ar'] ?? null,
                'logo' => $logoPath,
                'cover_image' => $coverPath,
                'status' => 'pending',
                'subscription_status' => 'inactive',
                'total_venues' => 0,
                'total_staff' => 0,
                'total_offers' => 0,
                'rating' => 0,
                'total_reviews' => 0,
            ]);

            // Assign business role to user
            $businessRole = Role::where('name', 'business_owner')->first();
            if ($businessRole) {
                $user->roles()->attach($businessRole->id, [
                    'business_id' => $business->id
                ]);
            }
//            Assign permission to user

            DB::commit();

            // Log the user in
            Auth::login($user);

            return redirect()->route('business.dashboard')
                ->with('success', 'Business registered successfully! Your application is pending verification.');

        } catch (\Exception $e) {
            DB::rollBack();

            return back()
                ->withErrors(['error' => 'Registration failed: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Check business name availability (AJAX)
     */
    public function checkAvailability(Request $request)
    {
        $businessExists = Business::where('business_name', $request->business_name)->exists();
        $licenseExists = Business::where('trade_license_number', $request->license)->exists();

        return response()->json([
            'business_name_available' => !$businessExists,
            'license_available' => !$licenseExists,
        ]);
    }
}
