<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Models\BusinessStaff;
use App\Models\Permission;
use App\Models\User;
use Illuminate\Http\Request;

class StaffController extends Controller
{
    /**
     * Display a listing of business staff members.
     */
    public function index(Request $request)
    {
        // Check if user is business owner or has permission
        if (!auth()->user()->isBusinessOwner() && !auth()->user()->hasTeamPermission('manage_staff')) {
            abort(403, 'Unauthorized');
        }

        // Get user's business
        $business = auth()->user()->businesses()->first();

        if (!$business) {
            abort(403, 'User does not own a business');
        }

        $query = $business->staff()->with('user', 'team');

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by role
        if ($request->filled('role')) {
            $query->where('role', $request->role);
        }

        // Search
        if ($request->filled('search')) {
            $query->whereHas('user', function ($q) {
                $q->where('first_name', 'like', '%' . request('search') . '%')
                    ->orWhere('last_name', 'like', '%' . request('search') . '%')
                    ->orWhere('email', 'like', '%' . request('search') . '%');
            });
        }

        $staff = $query->paginate(15);

        return view('business.staff.index', compact('business', 'staff'));
    }

    /**
     * Show the form for creating new staff.
     */
    public function create()
    {
        // Check if user is business owner or manager
        if (!auth()->user()->isBusinessOwner() && !auth()->user()->hasTeamPermission('manage_staff')) {
            abort(403, 'Unauthorized');
        }

        // Get user's business
        $business = auth()->user()->businesses()->first();

        if (!$business) {
            abort(403, 'User does not own a business');
        }

        $teams = $business->teams;

        // Get only important permissions (offers, venues, spotlight, redemption)
        $importantPermissions = Permission::whereIn('name', [
            'manage_offers',
            'manage_venues',
            'manage_spotlight',
            'process_redemptions'
        ])->get();

        return view('business.staff.create', compact('business', 'teams', 'importantPermissions'));
    }

    /**
     * Store a newly created staff member in storage.
     */
    public function store(Request $request)
    {
        // Check if user is business owner or manager
        if (!auth()->user()->isBusinessOwner() && !auth()->user()->hasTeamPermission('manage_staff')) {
            abort(403, 'Unauthorized');
        }

        // Get user's business
        $business = auth()->user()->businesses()->first();

        if (!$business) {
            abort(403, 'User does not own a business');
        }

        // Check subscription plan staff limit
        $subscription = $business->activeSubscription()->with('plan')->first();

        if (!$subscription) {
            return redirect()->back()
                ->with('error', 'No active subscription found. Please upgrade your plan to add staff.');
        }

        $maxStaff = $subscription->plan->max_staff;
        $currentStaff = $business->staff()->where('status', 'active')->count();

        if ($currentStaff >= $maxStaff) {
            return redirect()->back()
                ->with('error', 'You have reached the maximum staff limit (' . $maxStaff . ') for your current plan. Please upgrade to add more staff.');
        }

        $validated = $request->validate([
            'email' => 'required|email|unique:users,email',
            'role' => 'required|in:manager,staff',
            'team_id' => 'nullable|exists:teams,id',
            'permissions' => 'nullable|array',
            'permissions.*' => 'integer|exists:permissions,id',
        ]);

        // Create staff invitation record
        $staff = BusinessStaff::create([
            'business_id' => $business->id,
            'role' => $validated['role'],
            'team_id' => $validated['team_id'] ?? null,
            'status' => 'pending_invitation',
            'permissions' => !empty($validated['permissions']) ? json_encode($validated['permissions']) : null,
            'invited_at' => now(),
            'invitation_token' => \Str::random(60),
        ]);

        // Store email in BusinessStaff for invitation form display
        // Also cache it as backup
        $staff->update(['email' => $validated['email']]);
        \Cache::put('staff_invitation_email_' . $staff->invitation_token, $validated['email'], now()->addDays(7));

        // Send invitation email
        \Mail::send('emails.staff-invitation', [
            'business' => $business,
            'email' => $validated['email'],
            'role' => $validated['role'],
            'invitation_token' => $staff->invitation_token,
            'invitation_link' => route('staff.accept-invitation', $staff->invitation_token),
        ], function ($message) use ($validated, $business) {
            $message->to($validated['email'])
                ->subject("You're invited to join {$business->business_name} as a team member");
        });

        // Send notification to managers about new staff invitation
        $channels = config_value('notifications.channels.staff_invited', ['database']);
        foreach ($business->managers as $manager) {
            notify($manager->user, 'staff_invited', [
                'staff_email' => $validated['email'],
                'business_name' => $business->business_name,
                'role' => $validated['role'],
                'action_url' => route('business.staff.index'),
            ], $channels);
        }

        return redirect()->route('business.staff.index')
            ->with('success', 'Invitation sent to ' . $validated['email'] . '. (' . ($currentStaff + 1) . '/' . $maxStaff . ' staff members)');
    }

    /**
     * Show the form for editing staff member.
     */
    public function edit(BusinessStaff $staff)
    {
        // Check if user is business owner or manager
        if (!auth()->user()->isBusinessOwner() && !auth()->user()->hasTeamPermission('manage_staff')) {
            abort(403, 'Unauthorized');
        }

        // Get user's business
        $business = auth()->user()->businesses()->first();

        if (!$business || $staff->business_id !== $business->id) {
            abort(404, 'Staff member not found');
        }

        $teams = $business->teams;
        $permissions = Permission::all();

        return view('business.staff.edit', compact('business', 'staff', 'teams', 'permissions'));
    }

    /**
     * Update the specified staff member in storage.
     */
    public function update(Request $request, BusinessStaff $staff)
    {
        // Check if user is business owner or manager
        if (!auth()->user()->isBusinessOwner() && !auth()->user()->hasTeamPermission('manage_staff')) {
            abort(403, 'Unauthorized');
        }

        // Get user's business
        $business = auth()->user()->businesses()->first();

        if (!$business || $staff->business_id !== $business->id) {
            abort(404, 'Staff member not found');
        }

        $validated = $request->validate([
            'role' => 'required|in:owner,manager,staff',
            'team_id' => 'nullable|exists:teams,id',
        ]);

        $staff->update($validated);

        return redirect()->route('business.staff.index')
            ->with('success', 'Staff member updated successfully.');
    }

    /**
     * Remove the specified staff member from storage.
     */
    public function destroy(BusinessStaff $staff)
    {
        // Check if user is business owner or manager
        if (!auth()->user()->isBusinessOwner() && !auth()->user()->hasTeamPermission('manage_staff')) {
            abort(403, 'Unauthorized');
        }

        // Get user's business
        $business = auth()->user()->businesses()->first();

        if (!$business || $staff->business_id !== $business->id) {
            abort(404, 'Staff member not found');
        }

        $staff->delete();

        return redirect()->route('business.staff.index')
            ->with('success', 'Staff member removed successfully.');
    }

    /**
     * Show invitation acceptance form
     */
//    public function acceptInvitation($token)
//    {
//        $staff = BusinessStaff::where('invitation_token', $token)
//            ->where('status', 'pending_invitation')
//            ->firstOrFail();
//
//        return view('business.staff.accept-invitation', compact('staff'));
//    }

    /**
     * Process invitation acceptance and create user account
     */
    public function confirmInvitation(Request $request, $token)
    {
        $staff = BusinessStaff::where('invitation_token', $token)
            ->where('status', 'pending_invitation')
            ->firstOrFail();

        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'password' => 'required|string|min:8|confirmed',
        ]);

        // Create user account
        $user = User::create([
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'email' => $staff->email,
            'password' => bcrypt($validated['password']),
            'account_type' => 'business',
            'status' => 'active',
        ]);

        // Link staff to user
        $staff->update([
            'user_id' => $user->id,
            'status' => 'active',
            'joined_at' => now(),
            'invitation_token' => null,
        ]);

        // Assign role to user
        $roleMap = [
            'manager' => 'business_manager',
            'staff' => 'business_staff',
        ];

        if (isset($roleMap[$staff->role])) {
            $user->assignRole($roleMap[$staff->role]);
        }

        return redirect()->route('login')
            ->with('success', 'Account created successfully! You can now login with your email and password.');
    }

    /**
     * Show staff invitation acceptance page
     */
    public function acceptInvitation($token)
    {
        // Find staff by invitation token
        $staff = BusinessStaff::where('invitation_token', $token)
            ->where('status', 'pending_invitation')
            ->first();

        if (!$staff) {
            return redirect('/')->with('error', 'Invitation not found or already used.');
        }

        $business = $staff->business;

        // Get email from database (stored during invitation)
        // Fall back to cache if email not in DB
        $email = $staff->email ?? \Cache::get('staff_invitation_email_' . $token);

        if (!$email) {
            return redirect('/')->with('error', 'Invitation email not found. Please request a new invitation.');
        }

        return view('business.staff.accept-invitation', [
            'staff' => $staff,
            'business' => $business,
            'email' => $email,
            'token' => $token,
        ]);
    }

    /**
     * Complete staff invitation by creating account
     */
    public function completeInvitation(Request $request, $token)
    {
        try {
            // Find staff by invitation token
            $staff = BusinessStaff::where('invitation_token', $token)
                ->where('status', 'pending_invitation')
                ->first();

            if (!$staff) {
                return redirect('/')->with('error', 'Invitation not found or already used.');
            }

            $business = $staff->business;

            // Get email from database (primary source), fall back to cache
            $email = $staff->email ?? \Cache::get('staff_invitation_email_' . $token);

            if (!$email) {
                \Log::warning('Staff invitation email not found', [
                    'token' => $token,
                    'staff_id' => $staff->id,
                ]);
                return redirect('/')->with('error', 'Invitation email not found. Please request a new invitation.');
            }

            // Validate user input
            $validated = $request->validate([
                'first_name' => 'required|string|max:255',
                'last_name' => 'required|string|max:255',
                'password' => 'required|string|min:8|confirmed',
            ]);

            // Create user account with email
            $user = User::create([
                'first_name' => $validated['first_name'],
                'last_name' => $validated['last_name'],
                'email' => $email,
                'password' => bcrypt($validated['password']),
                'account_type' => 'business',
                'status' => 'active',
            ]);

            // Link staff to user
            $staff->update([
                'user_id' => $user->id,
                'status' => 'active',
                'joined_at' => now(),
                'invitation_token' => null,
                'email' => null,  // Clear email after use
            ]);

            // Clear cached email
            \Cache::forget('staff_invitation_email_' . $token);

            // Assign role to user
            $roleMap = [
                'manager' => 'business_manager',
                'staff' => 'business_staff',
            ];

            if (isset($roleMap[$staff->role])) {
                $user->assignRole($roleMap[$staff->role]);
            }

            // Send notification to business owner
            $channels = config_value('notifications.channels.staff_joined', ['database']);
            notify($business->owner, 'staff_joined', [
                'staff_name' => $user->first_name . ' ' . $user->last_name,
                'business_name' => $business->business_name,
                'role' => $staff->role,
                'action_url' => route('business.staff.index'),
            ], $channels);

            \Log::info('Staff invitation completed successfully', [
                'user_id' => $user->id,
                'staff_id' => $staff->id,
                'business_id' => $business->id,
            ]);

            return redirect()->route('login')
                ->with('success', 'Account created successfully! You can now login with your email and password.');
        } catch (\Exception $e) {
            \Log::error('Error completing staff invitation', [
                'token' => $token,
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
            ]);
            return redirect('/')->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }
}
