<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\BusinessSubscription;
use App\Models\SubscriptionPlan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SubscriptionController extends Controller
{
    /**
     * Display current subscription and usage
     */
    public function index()
    {
        $business = auth()->user()->business;

        if (!$business) {
            return redirect()->route('business.register');
        }

        $currentSubscription = $business->currentBusinessSubscription;
        $usage = $business->getSubscriptionUsage();
        $availablePlans = SubscriptionPlan::where('type', 'business')
            ->where('is_active', true)
            ->get();

        return view('business.subscription.index', compact('currentSubscription', 'usage', 'availablePlans'));
    }

    /**
     * Show available plans
     */
    public function plans(Request $request)
    {
        // Get plan type from query parameter, default to 'business'
        $planType = $request->query('type', 'business');

        // Validate plan type
        if (!in_array($planType, ['business', 'marketing', 'user'])) {
            $planType = 'business';
        }

        $plans = SubscriptionPlan::where('type', $planType)
            ->where('is_active', true)
            ->orderBy('sort_order', 'asc')
            ->orderBy('price', 'asc')
            ->get();

        $currentSubscription = auth()->user()->business->currentBusinessSubscription;

        // Get page title based on plan type
        $pageTitle = match($planType) {
            'marketing' => 'Marketing Plans',
            'user' => 'User Plans',
            default => 'Business Plans',
        };

        $pageDescription = match($planType) {
            'marketing' => 'Choose a marketing plan to create promotional campaigns',
            'user' => 'End-user subscription plans',
            default => 'Choose a business plan to manage venues and offers',
        };

        return view('business.subscription.plans', compact('plans', 'currentSubscription', 'planType', 'pageTitle', 'pageDescription'));
    }

    /**
     * Show plan details
     */
    public function showPlan($tier)
    {
        $plan = SubscriptionPlan::where('type', 'business')
            ->where('tier', $tier)
            ->where('is_active', true)
            ->firstOrFail();

        return view('business.subscription.plan-details', compact('plan'));
    }

    /**
     * Select and prepare payment for plan
     */
    public function selectPlan(Request $request, $tier)
    {
        $plan = SubscriptionPlan::where('type', 'business')
            ->where('tier', $tier)
            ->where('is_active', true)
            ->firstOrFail();

        // Redirect to upgrade method which handles both free and paid plans
        return redirect()->route('business.subscription.upgrade.plan', ['plan' => $plan->id]);
    }


    /**
     * Upgrade to a new plan (process payment)
     */
    public function upgrade(Request $request, $planId)
    {
        $business = auth()->user()->business ?? auth()->user()->businessStaff?->business;

        if (!$business) {
            return redirect()->route('business.register');
        }

        $plan = SubscriptionPlan::findOrFail($planId);

        // Only allow business plan type
        if ($plan->type !== 'business') {
            return redirect()->back()->with('error', 'Invalid plan selected');
        }

        // If plan is free (price = 0), directly activate subscription
        if ($plan->price == 0) {
            try {
                $this->activateBusinessSubscription($business, $plan);

                return redirect()->route('business.subscription.index')
                    ->with('success', 'Business plan activated successfully! You now have access to ' . $plan->name . ' including ' . ($plan->monthly_campaigns ?? 0) . ' promotional campaigns per month.');
            } catch (\Exception $e) {
                \Log::error('Subscription activation error', [
                    'error' => $e->getMessage(),
                    'plan_id' => $planId,
                    'business_id' => $business->id,
                ]);
                return redirect()->back()->with('error', 'Failed to activate subscription. Please try again.');
            }
        }

        // For paid plans, store in session and redirect to payment processing
        session(['upgrade_plan' => $planId]);
        session(['upgrade_business' => $business->id]);

        return redirect()->route('payment.checkout', ['plan_id' => $planId, 'type' => 'business']);
    }

    /**
     * Activate business subscription
     */
    private function activateBusinessSubscription($business, $plan)
    {
        // Only cancel the CURRENT active subscription (not all previous ones)
        $currentActiveSubscription = BusinessSubscription::where('business_id', $business->id)
            ->where('status', 'active')
            ->where('end_date', '>=', now())
            ->latest()
            ->first();

        if ($currentActiveSubscription) {
            $currentActiveSubscription->update([
                'status' => 'cancelled',
                'cancelled_at' => now()
            ]);
        }

        // Create new business subscription
        return BusinessSubscription::create([
            'business_id' => $business->id,
            'subscription_plan_id' => $plan->id,
            'status' => 'active',
            'start_date' => now(),
            'end_date' => now()->addDays($plan->duration_days ?? 30),
            'next_billing_date' => now()->addDays($plan->duration_days ?? 30),
            'auto_renew' => true,
            'payment_method' => 'free',
        ]);
    }

    /**
     * Cancel subscription
     */
    public function cancel(Request $request)
    {
        $business = auth()->user()->business ?? auth()->user()->businessStaff?->first()?->business;
        $subscription = $business->currentBusinessSubscription;

        if (!$subscription) {
            return response()->json([
                'success' => false,
                'message' => 'No active subscription found',
            ], 404);
        }

        $subscription->update([
            'status' => 'cancelled',
            'cancelled_at' => now(),
            'cancellation_reason' => $request->reason ?? null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Subscription cancelled successfully. Access will continue until ' . $subscription->end_date->format('M d, Y'),
        ]);
    }

    /**
     * Pause subscription
     */
    public function pause(Request $request)
    {
        $business = auth()->user()->business ?? auth()->user()->businessStaff?->first()?->business;
        $subscription = $business->currentBusinessSubscription;

        if (!$subscription || $subscription->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot pause this subscription',
            ], 400);
        }

        $subscription->update(['status' => 'paused']);

        return response()->json([
            'success' => true,
            'message' => 'Subscription paused',
        ]);
    }

    /**
     * Resume subscription
     */
    public function resume(Request $request)
    {
        $business = auth()->user()->business ?? auth()->user()->businessStaff?->first()?->business;
        $subscription = $business->currentBusinessSubscription;

        if (!$subscription || $subscription->status !== 'paused') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot resume this subscription',
            ], 400);
        }

        $subscription->update(['status' => 'active']);

        return response()->json([
            'success' => true,
            'message' => 'Subscription resumed',
        ]);
    }

    /**
     * Get subscription usage details
     */
    public function usage()
    {
        $business = auth()->user()->business ?? auth()->user()->businessStaff?->first()?->business;
        $usage = $business->getSubscriptionUsage();

        return response()->json([
            'success' => true,
            'usage' => $usage,
        ]);
    }

    /**
     * Renew subscription
     */
    public function renew(Request $request)
    {
        $business = auth()->user()->business ?? auth()->user()->businessStaff?->first()?->business;
        $subscription = $business->currentBusinessSubscription;

        if (!$subscription) {
            return response()->json([
                'success' => false,
                'message' => 'No active subscription found',
            ], 404);
        }

        // Redirect to payment for renewal
        session(['renew_subscription' => $subscription->id]);

        return response()->json([
            'success' => true,
            'redirect' => route('payment.renew', ['subscription_id' => $subscription->id]),
        ]);
    }

    /**
     * Get invoice
     */
    public function invoice($invoiceId)
    {
        $business = auth()->user()->business;

        // TODO: Generate PDF invoice

        return response()->json([
            'success' => true,
            'message' => 'Invoice generation not yet implemented',
        ]);
    }
}

